/*
 * Copyright (c) KylinSoft Co., Ltd. 2024. All rights reserved.
 *
 * kaiming is licensed under the GPL v2.0+.
 *
 * See the LICENSE file for more details.
 */

/**
 * @brief 开明穿透系统代理
 */

#include "common/KMCommonUtils.h"
#include "dbus/kmsystemproxy.h"

#define KAIMING_DBUS_SERVICE "org.kaiming.proxy"
#define KAIMING_DBUS_PATH "/org/kaiming/proxy"
#define KAIMING_DBUS_INTERFACE "org.kaiming.proxy.system"

struct StartBinaryTask
{
    int userId;
    std::string command;
};

class KMSystemProxy
{
public:
    static void onStartBinaryAsync(GTask *task, gpointer, gpointer taskData, GCancellable *)
    {
        auto *data = static_cast<StartBinaryTask *>(taskData);
        kmlogger.debug("[DBus] system uid: %d, command: %s", data->userId, data->command.c_str());
        runGtaskCommand(data->command, task);
    }

    static void onStartBinaryDone(GObject *, GAsyncResult *res, gpointer userData)
    {
        GError *error = nullptr;
        GTask *task = G_TASK(res);
        GDBusMethodInvocation *invocation = G_DBUS_METHOD_INVOCATION(g_task_get_source_object(task));
        gchar *result = static_cast<gchar *>(g_task_propagate_pointer(task, &error));
        if (error)
        {
            g_dbus_method_invocation_return_gerror(invocation, error);
            g_error_free(error);
        }
        else
        {
            proxy_system_complete_start_binary_as_user(PROXY_SYSTEM(userData), invocation, result);
            g_free(result);
        }
    }

    static gboolean onHandleStartBinaryAsUser(ProxySystem *, GDBusMethodInvocation *invocation, gint uid, const gchar *binary, gpointer userData)
    {
        auto *data = new StartBinaryTask{ uid, binary };
        auto *task = g_task_new(invocation, nullptr, onStartBinaryDone, userData);
        g_task_set_task_data(task, data, [](gpointer d) {
            delete static_cast<StartBinaryTask *>(d);
        });
        g_task_run_in_thread(task, onStartBinaryAsync);
        return TRUE;
    }
};

static void onBusAcquired(GDBusConnection *connection, const gchar *, gpointer)
{
    GError *error = nullptr;
    auto *service = proxy_system_skeleton_new();
    g_signal_connect(service, "handle-start-binary-as-user", G_CALLBACK(KMSystemProxy::onHandleStartBinaryAsUser), nullptr);
    g_dbus_interface_skeleton_export(G_DBUS_INTERFACE_SKELETON(service), connection, KAIMING_DBUS_PATH, &error);
    if (error)
    {
        g_clear_error(&error);
        g_object_unref(service);
        KMError("Failed to export kaiming system proxy interface .");
    }
}

int main()
{
    auto ownerId = g_bus_own_name(G_BUS_TYPE_SYSTEM, KAIMING_DBUS_SERVICE, G_BUS_NAME_OWNER_FLAGS_NONE, onBusAcquired, nullptr, nullptr, nullptr, nullptr);
    if (ownerId == 0)
    {
        KMError("Failed to acquire system dbus name .");
        return EXIT_FAILURE;
    }

    auto *loop = g_main_loop_new(nullptr, FALSE);
    if (!loop)
    {
        KMError("Failed to create GMainLoop .");
        g_bus_unown_name(ownerId);
        return EXIT_FAILURE;
    }

    g_main_loop_run(loop);
    g_bus_unown_name(ownerId);
    g_main_loop_unref(loop);
    return 0;
}