/*
 * Copyright (c) KylinSoft Co., Ltd. 2024. All rights reserved.
 *
 * kaiming is licensed under the GPL v2.0+.
 *
 * See the LICENSE file for more details.
 */
/**
 * @brief 开明穿透系统代理
 */

#include <vector>
#include <memory>

#include "common/KMCommonUtils.h"
#include "dbus/kmsessionproxy.h"

// D-Bus 名称和路径定义
#define KAIMING_DBUS_SERVICE "org.kaiming.proxy"
#define KAIMING_DBUS_PATH "/org/kaiming/proxy"
#define KAIMING_DBUS_INTERFACE "org.kaiming.proxy.session"

struct StartBinaryTask
{
    int userId;
    std::string command;
};

class KMSessionProxy
{
public:
    KMSessionProxy() = default;
    ~KMSessionProxy() = default;

    static void on_handle_xdg_open(ProxySession *object, GDBusMethodInvocation *invocation, const gchar *arg_name, gpointer user_data)
    {
        g_autofree gchar *local_path = nullptr;
        if (g_str_has_prefix(arg_name, "file://"))
        {
            local_path = g_filename_from_uri(arg_name, nullptr, nullptr);
            if (local_path == nullptr)
            {
                g_dbus_method_invocation_return_dbus_error(invocation, "org.kylin.kaiming.Error.Failed", "Invalid file:// URI.");
                KMError("Invalid file:// URI");
                return;
            }
        }
        else
        {
            local_path = g_strdup(arg_name);
        }

        GError *error = NULL;
        if (g_str_has_prefix(local_path, "http://") || g_str_has_prefix(local_path, "https://"))
        {
            g_autofree gchar *command = g_strdup_printf("xdg-open \"%s\"", local_path);
            if (!g_spawn_command_line_async(command, &error))
            {
                gchar *error_message = g_strdup_printf("Failed to open URL using xdg-open: %s", error->message ? error->message : "Unknown error");
                g_dbus_method_invocation_return_dbus_error(invocation, "org.kaiming.proxy.Error.Failed", error_message);
                KMError(error_message);
                g_free(error_message);
                g_error_free(error);
                return;
            }
            proxy_session_complete_xdg_open(object, invocation);
            kmlogger.info("Opened URL via xdg-open: %s", local_path);
            return;
        }

        if (g_file_test(local_path, G_FILE_TEST_EXISTS))
        {
            g_autofree gchar *command = g_strdup_printf("xdg-open \"%s\"", local_path);
            if (!g_spawn_command_line_async(command, &error))
            {
                gchar *error_message = g_strdup_printf("Failed to open file/path using xdg-open: %s", error->message ? error->message : "Unknown error");
                g_dbus_method_invocation_return_dbus_error(invocation, "org.kylin.kaiming.Error.Failed", error_message);
                KMError(error_message);
                g_free(error_message);
                g_error_free(error);
                return;
            }

            proxy_session_complete_xdg_open(object, invocation);
            kmlogger.info("Opened file/path via xdg-open: %s", local_path);
            return;
        }

        g_dbus_method_invocation_return_dbus_error(invocation, "org.kylin.kaiming.Error.Failed", "Invalid path or file does not exist.");
        KMError("Invalid path or file does not exist");
    }

    static void onStartBinaryAsync(GTask *task, gpointer, gpointer taskData, GCancellable *)
    {
        auto *data = static_cast<StartBinaryTask *>(taskData);
        kmlogger.debug("[DBus] session uid: %d, command: %s", data->userId, data->command.c_str());
        runGtaskCommand(data->command, task);
    }

    static void onStartBinaryDone(GObject *, GAsyncResult *res, gpointer userData)
    {
        GError *error = nullptr;
        GTask *task = G_TASK(res);
        GDBusMethodInvocation *invocation = G_DBUS_METHOD_INVOCATION(g_task_get_source_object(task));
        gchar *result = static_cast<gchar *>(g_task_propagate_pointer(task, &error));
        if (error)
        {
            g_dbus_method_invocation_return_gerror(invocation, error);
            g_error_free(error);
        }
        else
        {
            proxy_session_complete_start_binary_as_user(PROXY_SESSION(userData), invocation, result);
            g_free(result);
        }
    }

    static gboolean onHandleStartBinaryAsUser(KMSessionProxy *, GDBusMethodInvocation *invocation, gint uid, const gchar *binary, gpointer userData)
    {
        auto *data = new StartBinaryTask{ uid, binary };
        GTask *task = g_task_new(invocation, nullptr, onStartBinaryDone, userData);
        g_task_set_task_data(task, data, [](gpointer d) {
            delete static_cast<StartBinaryTask *>(d);
        });
        g_task_run_in_thread(task, onStartBinaryAsync);
        return TRUE;
    }
};

// 当总线获取成功时的回调
static void on_bus_acquired(GDBusConnection *connection, const gchar *name, gpointer user_data)
{
    GError *error = NULL;
    auto *service = proxy_session_skeleton_new();
    auto *handler = static_cast<KMSessionProxy *>(user_data);
    g_signal_connect(service, "handle-xdg-open", G_CALLBACK(KMSessionProxy::on_handle_xdg_open), handler);
    g_signal_connect(service, "handle-start-binary-as-user", G_CALLBACK(KMSessionProxy::onHandleStartBinaryAsUser), nullptr);
    g_dbus_interface_skeleton_export(G_DBUS_INTERFACE_SKELETON(service), connection, KAIMING_DBUS_PATH, &error);
    if (error)
    {
        KMError("Failed to export kaiming session proxy interface .");
        g_clear_error(&error);
        g_object_unref(service);
    }
}

int main(int argc, char **argv)
{
    KMSessionProxy handler;
    auto ownerId = g_bus_own_name(G_BUS_TYPE_SESSION, KAIMING_DBUS_SERVICE, G_BUS_NAME_OWNER_FLAGS_NONE, on_bus_acquired, NULL, NULL, &handler, NULL);
    if (ownerId == 0)
    {
        KMError("Failed to acquire system dbus name .");
        return EXIT_FAILURE;
    }

    auto *loop = g_main_loop_new(NULL, FALSE);
    if (!loop)
    {
        KMError("Failed to create GMainLoop .");
        g_bus_unown_name(ownerId);
        return EXIT_FAILURE;
    }

    g_main_loop_run(loop);
    g_bus_unown_name(ownerId);
    g_main_loop_unref(loop);
    return 0;
}