/*
 * Copyright (c) KylinSoft  Co., Ltd. 2024. All rights reserved.
 *
 * kaiming is licensed under the GPL v2.0+.
 * 
 * See the LICENSE file for more details.
 */

#include <fnmatch.h>
#include "KMParseSummaryJson.h"

#include "common/KMLogger.h"
#include "common/KMInstalledAppQuery.h"
#include "common/KMVersion.h"
#include "common/KMFormatter.h"

class KMParseSummaryJson::Private
{
public:
    Private();
    ~Private();

    json m_jsonData;
    bool m_isJsonLoaded = false;
    SummaryInfo m_summaryInfo;
    string completeRef;
    std::shared_ptr<KMFormatter> m_format;
    bool m_isThereDbusService = true;
    std::string m_notInstallable;
    vector<std::string> m_notFind;
};

KMParseSummaryJson::Private::Private()
{
}

KMParseSummaryJson::Private::~Private()
{
}

KMParseSummaryJson::KMParseSummaryJson()
    : d(make_shared<Private>())
{
    d->m_format = shared_ptr<KMFormatter>(new KMFormatter());
}

KMParseSummaryJson::KMParseSummaryJson(const std::string &filePath)
    : d(make_shared<Private>())
{
    std::ifstream inputFile(filePath);

    if (inputFile.is_open())
    {
        std::string jsonContent((std::istreambuf_iterator<char>(inputFile)), std::istreambuf_iterator<char>());
        std::string cleanedJson = removeComments(jsonContent); // 移除注释
        std::cout << cleanedJson << std::endl;

        // 使用 nlohmann::json 解析
        d->m_jsonData = json::parse(cleanedJson);

        // d->m_jsonData = json::parse(inputFile); // 解析带注释的 json 文件会报错，需要直接赋值，否则会报错
        inputFile.close();
    }
    else
    {
        throw std::runtime_error("Unable to open json file");
    }
}

KMParseSummaryJson::~KMParseSummaryJson()
{
}

void KMParseSummaryJson::clearCache()
{
    d->m_isJsonLoaded = false;
}

std::string KMParseSummaryJson::removeComments(const std::string &jsonString)
{
    std::regex commentPattern(R"(\/\/[^\n]*|\/\*[\s\S]*?\*\/)");
    return std::regex_replace(jsonString, commentPattern, "");
}

KMErrorCodeEnum KMParseSummaryJson::loadFile(const std::string &filePath, bool forceReload)
{
    fs::path p(filePath);
    if (!fs::exists(p)) {
        clearCache();
        return KM_ERROR_LOAD_FILE_FAILED;
    }

    // 如果不强制重载，且文件已加载，则使用缓存
    if (!forceReload && d->m_isJsonLoaded)
    {
        kmlogger.info("using cached json data from %s", filePath.c_str());
        return KM_ERROR_NO;
    }

    std::ifstream inputFile(filePath);
    if (!inputFile.is_open())
    {
        clearCache();
        return KM_ERROR_LOAD_FILE_FAILED;
    }

    d->m_jsonData = json::parse(inputFile);
    gernerateKMPackage();
    inputFile.close();

    d->m_isJsonLoaded = true;
    kmlogger.debug("loadFile success!");
    return KM_ERROR_NO;
}

bool KMParseSummaryJson::findMatchMaxVesion(KMPMRef pmRef, string moduleKey, ModuleInfo moduleData, string &maxVersionStr)
{
    vector<string> versionVector; // 保存符合条件的版本号
    std::string version;

    if (pmRef.ref.isSpecialVersion != true && !pmRef.ref.version.empty() && pmRef.ref.version[0] != '-')
    {
        KMVersion ver = KMVersion::parse(pmRef.ref.version, false);
        version = std::to_string(ver.getMajor());
    }
    else
    {
        version = pmRef.ref.version;
    }

    for (const auto &[versionKey, versionData] : moduleData.versionInfo)
    {

        if (!pmRef.ref.version.empty() && pmRef.ref.version[0] != '-')
        {
            if (pmRef.ref.version != versionKey)
            {
                if (KMStringUtils::startsWith(versionKey, version))
                {
                    versionVector.push_back(versionKey);
                }
            }
            else
            {
                versionVector.push_back(versionKey);
            }
        }
        else
        {
            versionVector.push_back(versionKey);
        }
    }

    if (versionVector.size() > 0)
    {
        KMVersion maxVersion = KMVersion::parse(versionVector.at(0), false);
        maxVersionStr = versionVector.at(0);
        for (const auto &version : versionVector)
        {
            KMVersion ver = KMVersion::parse(version, false);

            if (maxVersion < ver)
            {
                maxVersion = ver;
                maxVersionStr = version;
            }
        }

        // maxVersionStr = maxVersion.str();
        return true;
    }
    else
    {
        return false;
    }
}

void KMParseSummaryJson::from_json(const json &j, VersionInfo &version)
{
    if (j.contains("installed-size"))
    {
        j.at("installed-size").get_to(version.installedSize);
    }

    if (j.contains("extensions"))
    {
        j.at("extensions").get_to(version.extensions);
    }

    if (j.contains("base"))
    {
        j.at("base").get_to(version.base);
    }

    if (j.contains("runtime"))
    {
        j.at("runtime").get_to(version.runtime);
    }
    if (j.contains("depends") && j.at("depends").is_array())
    {
        j.at("depends").get_to(version.depend);
    }
}

void KMParseSummaryJson::from_json(const json &j, ModuleInfo &module)
{
    for (auto &item : j.items())
    {
        if (KMStringUtils::startsWith(item.key(), "name"))
        {
            module.nameInfo[item.key()] = item.value();
        }
        else if (KMStringUtils::startsWith(item.key(), "description"))
        {
            module.descriptionInfo[item.key()] = item.value();
        }
        else
        {
            VersionInfo versionData;
            from_json(item.value(), versionData);
            module.versionInfo[item.key()] = versionData;
        }
    }
}

void KMParseSummaryJson::from_json(const json &j, IdInfo &id)
{
    for (auto &module : j.items())
    {
        ModuleInfo moduleData;
        from_json(module.value(), moduleData);
        id.moduleInfo[module.key()] = moduleData;
    }
}

void KMParseSummaryJson::from_json(const json &j, KindInfo &kind)
{
    for (auto &id : j.items())
    {
        IdInfo idData;
        from_json(id.value(), idData);
        kind.idInfo[id.key()] = idData;
    }
}

void KMParseSummaryJson::from_json(const json &j, ArchInfo &arch)
{
    for (auto &kind : j.items())
    {
        KindInfo kindData;
        from_json(kind.value(), kindData);
        arch.kindInfo[kind.key()] = kindData;
    }
}

void KMParseSummaryJson::from_json(const json &j, ChannelInfo &channel)
{
    for (auto &arch : j.items())
    {
        // a.archInfo[arch.key()] = arch.value().get<TypeInfo>(); // 此种方式不行
        ArchInfo archData;
        from_json(arch.value(), archData);
        channel.archInfo[arch.key()] = archData;
    }
}

void KMParseSummaryJson::from_json(const json &j, SummaryInfo &summary)
{
    for (const auto &[channelKey, channelValue] : j.items())
    {
        kmlogger.debug("arch: %s value: %s", channelKey.c_str(), channelValue.dump().c_str());

        ChannelInfo channelData;
        from_json(channelValue, channelData);
        summary.channelInfo.emplace(channelKey, std::move(channelData));
    }
}

void KMParseSummaryJson::printJson()
{
    kmlogger.debug("summary.json: %s", d->m_jsonData.dump(4).c_str());

    // 第一层遍历 Json 数据
    for (auto &item : d->m_jsonData.items())
    {
        kmlogger.debug("item: %s value: %s", item.key().c_str(), item.value());
    }
}

void KMParseSummaryJson::parseSummaryJson()
{
    from_json(d->m_jsonData, d->m_summaryInfo);
    printSummaryJson(d->m_summaryInfo);
}

void KMParseSummaryJson::gernerateKMPackage()
{
    printJson();
    parseSummaryJson();
    getPackagesFromSummary();
}

void KMParseSummaryJson::getPackagesFromSummary()
{
    SummaryInfo data = d->m_summaryInfo;
    fs::path packagesPath = "/opt/kaiming/packages";
    std::set<std::string> packageIds; //去重
    // 打开文件 写入模式，自动创建文件
    std::ofstream outfile(packagesPath);
    for (const auto &[channelKey, channelData] : data.channelInfo)
    {
        kmlogger.debug("channelKey: %s", channelKey.c_str()); // main, stable
        for (const auto &[archKey, archData] : channelData.archInfo)
        {
            kmlogger.debug("archKey: %s", archKey.c_str()); // x86_64, arm64
            for (const auto &[kindKey, kindData] : archData.kindInfo)
            {
                kmlogger.debug("kindKey: %s", kindKey.c_str()); // app, runtime, base, extension
                for (const auto &[idKey, idData] : kindData.idInfo)
                {
                    kmlogger.debug("idKey: %s", idKey.c_str()); // top.openkylin.Music, top.openkylin.base, top.ukui.runtime
                    packageIds.insert(idKey.c_str());
                }
            }
        }
    }

    for (const auto &id : packageIds)
    {
        outfile << id << std::endl;
    }
    outfile.close();
}

void KMParseSummaryJson::printSummaryJson(const SummaryInfo &data)
{
    for (const auto &[channelKey, channelData] : data.channelInfo)
    {
        kmlogger.debug("channelKey: %s", channelKey.c_str()); // main, stable
        for (const auto &[archKey, archData] : channelData.archInfo)
        {
            kmlogger.debug("archKey: %s", archKey.c_str()); // x86_64, arm64
            for (const auto &[kindKey, kindData] : archData.kindInfo)
            {
                kmlogger.debug("kindKey: %s", kindKey.c_str()); // app, runtime, base, extension
                for (const auto &[idKey, idData] : kindData.idInfo)
                {
                    kmlogger.debug("idKey: %s", idKey.c_str()); // top.openkylin.Music, top.openkylin.base, top.ukui.runtime
                    for (const auto &[moduleKey, moduleData] : idData.moduleInfo)
                    {
                        kmlogger.debug("moduleKey: %s", moduleKey.c_str());
                        {
                            for (const auto &[nameKey, nameData] : moduleData.nameInfo)
                            {
                                if (KMStringUtils::startsWith(nameKey, "name"))
                                {
                                    kmlogger.debug("nameKey: %s", nameKey.c_str());
                                }
                            }

                            for (const auto &[descriptionKey, descriptionData] : moduleData.descriptionInfo)
                            {
                                if (KMStringUtils::startsWith(descriptionKey, "name"))
                                {
                                    kmlogger.debug("descriptionKey: %s", descriptionKey.c_str());
                                }
                            }
                        }

                        {
                            for (const auto &[versionKey, versionData] : moduleData.versionInfo)
                            {
                                kmlogger.debug("versionKey: %s", versionKey.c_str()); // 2.0.0.0

                                // 打印每个 version
                                kmlogger.debug("installedSize: %ld base: %s runtime: %s", versionData.installedSize, versionData.base.empty() ? "" : versionData.base.c_str(),
                                               versionData.runtime.empty() ? "" : versionData.runtime.c_str());

                                if (!versionData.extensions.empty())
                                {
                                    kmlogger.debug(" Extension: ");
                                    for (const auto &ext : versionData.extensions)
                                    {
                                        if (!ext.empty())
                                        {
                                            kmlogger.debug("ext: %s", ext.c_str());
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}

string KMParseSummaryJson::toLowerUtf8(const std::string &str)
{
    wstring_convert<std::codecvt_utf8<wchar_t>> converter;
    wstring wstr = converter.from_bytes(str);
    transform(wstr.begin(), wstr.end(), wstr.begin(),
              [](wchar_t c)
              { return std::towlower(c); });
    return converter.to_bytes(wstr);
}

string KMParseSummaryJson::printSearchJson(const string &searchId, const string &specifyArch, const string &specifyKind, const string &specifyModule, bool isPrint, bool isGetDevel)
{
    if (isPrint)
    {
        if (specifyModule == "devel")
        {
            d->m_format->setShowBase(true);
            d->m_format->setShowRuntime(true);
        }
        d->m_format->setColumnWidth(45, 30, 25, 10, 15, 15, 30, 15, 25, 25);
        d->m_format->setShowArch(false);
        d->m_format->setShowChannel(true);
        d->m_format->printHeader();
    }
    d->completeRef.clear();
    KMEntry entry;
    KMUtil m_localeLanguage;
    int byteSize;
    std::string localeName = m_localeLanguage.getCurrentLocale();

    auto matchId = [](const std::string &pattern, const std::string &text) -> bool
    {
        if (pattern.find('*') != std::string::npos)
        {
            // 使用 fnmatch 进行通配符匹配
            return fnmatch(pattern.c_str(), text.c_str(), FNM_NOESCAPE) == 0;
        }
        else
        {
            return text.find(pattern) != std::string::npos;
        }
    };

    for (auto &channel : d->m_jsonData.items())
    {
        kmlogger.debug("channel: %s", channel.key().c_str());
        entry.channel = channel.key();
        for (auto &arch : channel.value().items())
        {
            kmlogger.debug("arch: %s", arch.key().c_str());
            entry.arch = arch.key();
            if (!specifyArch.empty() && arch.key() != specifyArch)
            {
                continue;
            }
            for (auto &kind : arch.value().items())
            {
                kmlogger.debug("kind: %s", kind.key().c_str());
                if (!specifyKind.empty() && kind.key() != specifyKind)
                {
                    continue;
                }
                for (auto &id : kind.value().items())
                {
                    string fid = id.key();

                    string fidLower = toLowerUtf8(fid);
                    string idLower = toLowerUtf8(searchId);
                    if (matchId(idLower, fidLower))
                    {
                        entry.id = fid;
                        for (auto &module : id.value().items())
                        {
                            entry.module = module.key();
                            kmlogger.debug("module: %s", module.key().c_str());
                            if (!specifyModule.empty() && module.key() != specifyModule)
                            {
                                continue;
                            }
                            if (module.value().contains("name") && module.value().contains("description"))
                            {
                                entry.name = !module.value()["name[" + localeName + "]"].empty() ? module.value()["name[" + localeName + "]"] : module.value()["name"];

                                entry.description = !module.value()["description[" + localeName + "]"].empty() ? module.value()["description[" + localeName + "]"] : module.value()["description"];
                                vector<string> versionVector;
                                for (auto &version : module.value().items())
                                {
                                    if (version.key() != "name" && version.key() != "description" && version.key() != "name[" + localeName + "]" && version.key() != "description[" + localeName + "]")
                                    {
                                        entry.version = version.key();
                                        if(searchId == id.key() && (entry.module == "binary" || isGetDevel))
                                        {
                                            versionVector.push_back(entry.version);
                                        }
                                        string installedSize = KMUtil::formatInstalledSize((double)version.value()["installed-size"]);
                                        byteSize = (double)version.value()["installed-size"];
                                        entry.size = installedSize;
                                        entry.base = version.value()["base"].get<std::string>();
                                        entry.runtime = version.value()["runtime"].get<std::string>();
                                        if (isPrint)
                                        {
                                            d->m_format->printEntry(entry);
                                        }
                                    }
                                }
                                if (versionVector.size() > 0)
                                {
                                    KMVersion maxVersion = KMVersion::parse(versionVector.at(0), true);
                                    for (const auto &version : versionVector)
                                    {
                                        KMVersion ver = KMVersion::parse(version, true);

                                        if (maxVersion < ver)
                                        {
                                            maxVersion = ver;
                                        }
                                    }
                                    d->completeRef = entry.channel;
                                    d->completeRef.append("/").append(entry.arch).append("/").append(kind.key()).append("/")
                                        // .append(entry.kind).append("/")
                                        .append(entry.id)
                                        .append("/")
                                        .append(entry.module)
                                        .append("/")
                                        .append(maxVersion.str())
                                        .append("/")
                                        .append(entry.size)
                                        .append("/")
                                        .append(std::to_string(byteSize))
                                        .append("/");
                                    versionVector.clear();
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    if(isPrint)
    {
        d->m_format->printFooter();
    }
    return d->completeRef;
}

bool KMParseSummaryJson::getSummaryInfo(KMPMRef &pmRef)
{
    string sortSearchRef;
    bool isSearched = false;

    for (const auto &[channelKey, channelData] : d->m_summaryInfo.channelInfo)
    {
        kmlogger.debug("channelKey: %s", channelKey.c_str());
        if (!pmRef.ref.channel.empty() && channelKey != pmRef.ref.channel)
        {
            continue;
        }

        for (const auto &[archKey, archData] : channelData.archInfo)
        {
            kmlogger.debug("archKey: %s", archKey.c_str());
            if (!pmRef.ref.arch.empty() && pmRef.ref.arch != archKey)
            {
                continue;
            }

            for (const auto &[kindKey, kindData] : archData.kindInfo)
            {
                kmlogger.debug("kindKey: %s", kindKey.c_str());
                if (!pmRef.ref.kind.empty() && pmRef.ref.kind != kindKey)
                {
                    continue;
                }

                for (const auto &[idKey, idData] : kindData.idInfo)
                {
                    kmlogger.debug("idKey: %s", idKey.c_str());
                    if (!pmRef.ref.id.empty() && pmRef.ref.id != idKey)
                    {
                        continue;
                    }

                    for (const auto &[moduleKey, moduleData] : idData.moduleInfo)
                    {
                        kmlogger.debug("moduleKey: %s", moduleKey.c_str());
                        if (!pmRef.ref.module.empty() && pmRef.ref.module != moduleKey)
                        {
                            continue;
                        }

                        // 获取 name 和 description 信息
                        getSummaryNameInfo(pmRef, moduleData);

                        {
                            string maxVersionStr;

                            bool getMaxVersionStrFlag = false;
                            getMaxVersionStrFlag = findMatchMaxVesion(pmRef, moduleKey, moduleData, maxVersionStr);

                            if (!getMaxVersionStrFlag)
                            {
                                return isSearched;
                            }

                            for (const auto &[versionKey, versionData] : moduleData.versionInfo)
                            {
                                kmlogger.debug("versionKey: %s", versionKey.c_str());
                                if (maxVersionStr != versionKey)
                                {
                                    continue;
                                }

                                isSearched = true;

                                pmRef.installedSize = versionData.installedSize;

                                // 打印每个module
                                kmlogger.debug("installedSize: %ld base: %s runtime: %s", versionData.installedSize, versionData.base.empty() ? "" : versionData.base.c_str(),
                                               versionData.runtime.empty() ? "" : versionData.runtime.c_str());
                            }
                        }
                    }
                }
            }
        }
    }

    return isSearched;
}

bool KMParseSummaryJson::getSummaryNameInfo(KMPMRef &pmRef, ModuleInfo moduleInfo)
{
    string locale = KMUtil::getCurrentLocale();
    string nameStr;
    string descriptionStr;

    for (const auto &[nameKey, nameData] : moduleInfo.nameInfo)
    {
        if (KMStringUtils::startsWith(nameKey, "name"))
        {
            kmlogger.debug("nameKey: %s", nameKey.c_str());
            if (nameKey == "name" || locale == "C")
            {
                nameStr = nameData;
            }
            else if (KMStringUtils::contains(nameKey, locale))
            {
                pmRef.name = nameData;
            }
        }
    }

    if (pmRef.name.empty())
    {
        pmRef.name = nameStr;
    }

    for (const auto &[descriptionKey, descriptionData] : moduleInfo.descriptionInfo)
    {
        if (KMStringUtils::startsWith(descriptionKey, "description"))
        {
            kmlogger.debug("descriptionKey: %s", descriptionKey.c_str());
            if (descriptionKey == "description" || locale == "C")
            {
                descriptionStr = descriptionData;
                KMStringUtils::removeTrailingNewline(descriptionStr);
            }
            else if (KMStringUtils::contains(descriptionKey, locale))
            {
                pmRef.description = descriptionData;
                KMStringUtils::removeTrailingNewline(pmRef.description);
            }
        }
    }

    if (pmRef.description.empty())
    {
        pmRef.description = descriptionStr;
    }

    return true;
}

bool KMParseSummaryJson::fuzzyMatchCompletedRef(KMPMRef pmRef, vector<string> &fuzzyMatchRefs)
{
    string searchRef;
    string sortSearchRef;
    bool isSearched = false;

    if (d->m_isThereDbusService)
    {
        const char *dbusmethod = "GetPackageInstallStatus";
        const char *idChar = pmRef.ref.id.c_str();
        const char *dbusInterface = "com.kylin.kysdk.applicationsec.antiinstall";
        int getDbusReturn = getDbus(dbusInterface, dbusmethod, idChar);
        if (getDbusReturn == -1)
        {
            d->m_isThereDbusService = false;
        }
        if (getDbusReturn == 1)
        {
            d->m_notInstallable = pmRef.ref.id;
            return false;
        }
    }

    for (const auto &[channelKey, channelData] : d->m_summaryInfo.channelInfo)
    {
        searchRef.clear();
        kmlogger.debug("channelKey: %s", channelKey.c_str());
        if (!pmRef.ref.channel.empty() && channelKey != pmRef.ref.channel)
        {
            continue;
        }
        string channelStr = searchRef.append(channelKey).append("/");

        for (const auto &[archKey, archData] : channelData.archInfo)
        {
            searchRef = channelStr;
            kmlogger.debug("archKey: %s", archKey.c_str());
            if (!pmRef.ref.arch.empty() && pmRef.ref.arch != archKey)
            {
                continue;
            }
            string archStr = searchRef.append(archKey).append("/");

            for (const auto &[kindKey, kindData] : archData.kindInfo)
            {
                searchRef = archStr;
                kmlogger.debug("kindKey: %s", kindKey.c_str());
                if (!pmRef.ref.kind.empty() && pmRef.ref.kind != kindKey)
                {
                    continue;
                }

                string kindStr = searchRef.append(kindKey).append("/");

                for (const auto &[idKey, idData] : kindData.idInfo)
                {
                    searchRef = kindStr;
                    kmlogger.debug("idKey: %s", idKey.c_str());
                    if (!pmRef.ref.id.empty() && pmRef.ref.id != idKey)
                    {
                        continue;
                    }
                    if (kindKey == "depend" && pmRef.ref.module.empty())
                    {
                        pmRef.ref.module = "devel";
                    }
                    string idStr = searchRef.append(idKey).append("/");

                    for (const auto &[moduleKey, moduleData] : idData.moduleInfo)
                    {
                        searchRef = idStr;
                        kmlogger.debug("moduleKey: %s", moduleKey.c_str());
                        if (!pmRef.ref.module.empty() && pmRef.ref.module != moduleKey)
                        {
                            continue;
                        }

                        // module 为空时, 匹配所有 binary 模块
                        if (pmRef.ref.module.empty() && moduleKey != "binary")
                        {
                            continue;
                        }

                        string moduleStr = searchRef.append(moduleKey).append("/");
                        pmRef.ref.kind = kindKey;

                        isSearched = matchVersionData(pmRef, moduleKey, moduleData, moduleStr, fuzzyMatchRefs);
                    }
                }
            }
        }
    }

    if (isSearched == false)
    {
        d->m_notFind.push_back(pmRef.ref.id);
    }

    return isSearched;
}

bool KMParseSummaryJson::matchVersionData(KMPMRef pmRef, string moduleKey, ModuleInfo moduleData, string searchRef, vector<string> &fuzzyMatchRefs)
{
    bool isSearched = false;
    string tmpSearchRef = searchRef;

    {
        string maxVersionStr;
        bool getMaxVersionStrFlag = false;
        getMaxVersionStrFlag = findMatchMaxVesion(pmRef, moduleKey, moduleData, maxVersionStr);

        if (!getMaxVersionStrFlag)
        {
            d->m_notFind.push_back(pmRef.ref.id);
            return isSearched;
        }

        for (const auto &[versionKey, versionData] : moduleData.versionInfo)
        {
            searchRef = tmpSearchRef;
            kmlogger.debug("versionKey: %s", versionKey.c_str());
            if (maxVersionStr != versionKey)
            {
                continue;
            }
            string versionStr = searchRef.append(versionKey);

            isSearched = true;

            bool isMatch = false;
            for (const auto &version : fuzzyMatchRefs)
            {
                if (version.find(versionStr) != std::string::npos)
                {
                    isMatch = true;
                    break;
                }
            }

            if (isMatch)
            {
                continue;
            }

            fuzzyMatchRefs.push_back(versionStr);
            // base 获取队列
            if (!versionData.base.empty())
            {
                KMPMRef pmBaseRef = KMUtil::parseSummaryKMPMRef(versionData.base);
                pmBaseRef.ref.kind = BASE_TYPE_BASE;
                pmBaseRef.ref.module = pmRef.ref.module;

                if (!currentFuzzyRefIsInstalled(pmBaseRef))
                {
                    if (!fuzzyMatchCompletedRef(pmBaseRef, fuzzyMatchRefs))
                    {
                        return false;
                    }
                }
            }

            // runtime 获取队列
            if (!versionData.runtime.empty())
            {
                KMPMRef pmRuntimeRef = KMUtil::parseSummaryKMPMRef(versionData.runtime);
                pmRuntimeRef.ref.kind = BASE_TYPE_RUNTIME;
                pmRuntimeRef.ref.module = pmRef.ref.module;

                if (!currentFuzzyRefIsInstalled(pmRuntimeRef))
                {
                    if (!fuzzyMatchCompletedRef(pmRuntimeRef, fuzzyMatchRefs))
                    {
                        return false;
                    }
                }
            }

            // depend 获取队列
            if (!versionData.depend.empty())
            {
                for (std::string dependStr : versionData.depend)
                {
                    KMPMRef pmDependRef;
                    std::vector<std::string> tokens;
                    std::stringstream ss(dependStr);
                    std::string token;
                    while (std::getline(ss, token, '/'))
                    {
                        tokens.push_back(token); // 将切分后的部分添加到 tokens 向量中
                    }
                    if (dependStr.find('/') != std::string::npos)
                    {
                        pmDependRef.ref.arch = pmRef.ref.arch;
                        pmDependRef.ref.id = tokens[0];
                        pmDependRef.ref.version = tokens[1];
                        pmDependRef.ref.module = pmRef.ref.module;
                        pmDependRef.ref.channel = pmRef.ref.channel;
                        if (!currentFuzzyRefIsInstalled(pmDependRef))
                        {
                            if (!fuzzyMatchCompletedRef(pmDependRef, fuzzyMatchRefs))
                            {
                                return false;
                            }
                        }
                    }
                    else
                    {
                        pmDependRef.ref.arch = pmRef.ref.arch;
                        pmDependRef.ref.id = tokens[0];
                        pmDependRef.ref.version = "-";
                        pmDependRef.ref.module = pmRef.ref.module;
                        pmDependRef.ref.channel = pmRef.ref.channel;
                        if (!currentFuzzyRefIsInstalled(pmDependRef))
                        {
                            if (!fuzzyMatchCompletedRef(pmDependRef, fuzzyMatchRefs))
                            {
                                return false;
                            }
                        }
                    }
                }
            }

            // 打印每个module
            kmlogger.debug("installedSize: %ld base: %s runtime: %s", versionData.installedSize, versionData.base.empty() ? "" : versionData.base.c_str(),
                           versionData.runtime.empty() ? "" : versionData.runtime.c_str());

            // extension 获取队列
            if (!versionData.extensions.empty())
            {
                kmlogger.debug(" Extension: ");
                for (const auto &ext : versionData.extensions)
                {
                    if (!ext.empty())
                    {
                        kmlogger.debug("ext: %s", ext.c_str());
                        KMPMRef pmExtensionRef = KMUtil::parseSummaryKMPMRef(ext);
                        pmExtensionRef.ref.kind = BASE_TYPE_EXTENSION;

                        if (!fuzzyMatchCompletedRef(pmExtensionRef, fuzzyMatchRefs))
                        {
                            return false;
                        }
                    }
                }
            }
        }
    }

    if (isSearched == false)
    {
        d->m_notFind.push_back(pmRef.ref.id);
    }

    return isSearched;
}

bool KMParseSummaryJson::currentFuzzyRefIsInstalled(KMPMRef &pmRef)
{
    KMInstalledAppQuery query;
    bool isInstalled = false;

    string kindStr = pmRef.ref.kind.empty() ? "" : pmRef.ref.kind;
    string versionStr = "";
    string channelStr = pmRef.ref.channel.empty() ? "" : pmRef.ref.channel;
    string moduleStr = pmRef.ref.module.empty() ? "binary" : pmRef.ref.module;

    std::vector<KMRef> queryVector = query.kind(kindStr).id(pmRef.ref.id).channel(channelStr).version(versionStr).module(moduleStr).query(KMInstalledAppQuery::All);

    KMVersion installedVersion;
    KMVersion currentVersion;

    if (!queryVector.empty())
    {
        if (pmRef.ref.version[0] == '-')
        {
            return isInstalled;
        }
        currentVersion = KMVersion::parse(pmRef.ref.version, false);

        // 指定版本，则检查是否已安装
        for (const auto &ref : queryVector)
        {
            installedVersion = KMVersion::parse(ref.version, false);
            if (ref.kind == "base" || ref.kind == "runtime" || ref.kind == "depend")
            {
                if (installedVersion.getMajor() == currentVersion.getMajor() && installedVersion.getMinor() == currentVersion.getMinor())
                {
                    if (installedVersion.getPatch() >= currentVersion.getPatch())
                    {
                        isInstalled = true;
                        break;
                    }
                }
            }
        }
    }
    return isInstalled;
}

int KMParseSummaryJson::getDbus(const char *interfaceName, const char *methodName, const char *package)
{
    GError *error = NULL;

    // 1. 连接到系统总线
    GDBusConnection *connection = g_bus_get_sync(G_BUS_TYPE_SYSTEM, NULL, &error);
    if (!connection)
    {
        kmlogger.error("D-Bus connection failed: %s", error->message);
        g_clear_error(&error);
        return -1;
    }

    // 2. 检查服务是否存在
    GVariant *services = g_dbus_connection_call_sync(
        connection,
        "org.freedesktop.DBus",
        "/org/freedesktop/DBus",
        "org.freedesktop.DBus",
        "ListNames",
        NULL,
        G_VARIANT_TYPE("(as)"),
        G_DBUS_CALL_FLAGS_NONE,
        -1,
        NULL,
        &error);

    if (error)
    {
        kmlogger.error("Failed to list DBus services: %s", error->message);
        g_clear_error(&error);
        g_object_unref(connection);
        return -1;
    }

    // 3. 检查目标服务是否在列表中
    GVariantIter *iter;
    g_variant_get(services, "(as)", &iter);
    gboolean service_found = FALSE;
    const char *service_name;

    while (g_variant_iter_next(iter, "s", &service_name))
    {
        if (g_strcmp0(service_name, "com.kylin.kysdk.applicationsec") == 0)
        {
            service_found = TRUE;
            break;
        }
    }
    g_variant_iter_free(iter);
    g_variant_unref(services);

    if (!service_found)
    {
        kmlogger.error("DBus service 'com.kylin.kysdk.applicationsec' not found");
        g_object_unref(connection);
        return -1;
    }

    // 4. 调用目标方法
    GVariant *params = g_variant_new("(s)", package);
    GVariant *result = g_dbus_connection_call_sync(
        connection,
        "com.kylin.kysdk.applicationsec",
        "/com/kylin/kysdk/applicationsec",
        interfaceName,
        methodName,
        params,
        G_VARIANT_TYPE("(i)"),
        G_DBUS_CALL_FLAGS_NONE,
        -1,
        NULL,
        &error);

    // 5. 处理调用结果
    int ret = 1; // 默认禁止安装
    if (error)
    {
        gint32 response_value;
        kmlogger.error("D-Bus call '%s.%s' failed: %s", interfaceName, methodName, error->message);
        g_variant_get(result, "(i)", &response_value);
        ret = response_value;
        g_clear_error(&error);
    }
    else if (!result)
    {
        kmlogger.error("D-Bus call returned NULL");
    }
    else
    {
        gint32 response_value;
        g_variant_get(result, "(i)", &response_value);
        kmlogger.debug("D-Bus call returned: %d", response_value);
        ret = response_value;
        g_variant_unref(result);
    }

    g_object_unref(connection);
    return ret;
}

std::string KMParseSummaryJson::getNotInstallable()
{
    return d->m_notInstallable;
}

vector<std::string> KMParseSummaryJson::getNotFind()
{
    return d->m_notFind;
}

bool KMParseSummaryJson::clearNotInstallable()
{
    d->m_notFind.clear();
    d->m_notInstallable = "";
    return true;
}