/*
 * Copyright (c) KylinSoft  Co., Ltd. 2024. All rights reserved.
 *
 * kaiming is licensed under the GPL v2.0+.
 * 
 * See the LICENSE file for more details.
 */

#include <sys/ioctl.h>
#include <unistd.h>
#include <thread>
#include <atomic>
#include <string>
#include <iostream>
#include <sstream>
#include <dlfcn.h>

#include "KMPackageManager.h"

#include "common/KMLogger.h"
#include "common/KMInstalledAppQuery.h"
#include "common/KMDbusJson.h"
#include "common/KMVersion.h"
#include "common/KMStringUtils.h"
#include "common/KMFileUtils.h"
#include "common/KMLibHandle.h"
#include "common/KMDirMd5sum.h"
#include "app/list/KMList.h"

#define TEST_ONLINE 1

static vector<KMPMRef> globalRefDownloadListForProgress;         // 实际上需要安装的软件列表,用于进度展示
static KMPMProgressCallBack globalProgressCallback;              // 回调函数
static KMPMProgressStatus globalProgressStatus = {"", 0, "", 5}; // 安装进度回调函数参数
static string progressJson = "";

static string lastOutput;        // 保存最后一次进度条输出
static string preRemaining = ""; // 保存先前的剩余时间

static int currentPercent = 0; // 当前进度

static int lastWidth = 0;
static std::string lastProgressStr;
static std::string lastStaticInfo;
static bool staticInfoDisplayed = false;

std::atomic<int> sharedValue(0); // 定义一个原子类型的共享变量，初始值为0

std::once_flag KMPackageManager::initFlag;
std::vector<std::string> KMPackageManager::drivers;
constexpr int SUMMARY_EXPIRE_SECONDS = 3600; // the expire check time of summary

class KMPackageManager::Private
{
public:
    std::shared_ptr<KMFolder> m_folder;
    std::shared_ptr<KMList> m_list;
    std::shared_ptr<KMCurl> m_curl;
    std::shared_ptr<KMUtil> m_util;
    std::shared_ptr<KMEnv> m_env;
    std::shared_ptr<KMParseSummaryJson> m_parseSummaryJson; // 解析 summary.json
    std::shared_ptr<KMFormatter> m_formatter;               // 格式化输出
    string m_repoPath;                                      // 仓库路径
    bool m_pmIsForceInstall = false;                        // 是否强制安装
    bool m_pmIsThereDbusService = true;                     // 是否有dbus安装卸载检测服务
    bool m_isHostapp = false;                               // 是否是hostapp
    KMPMRef m_pmRef;                                        // 单独ref,用于存储到 d->m_refUserList中
    vector<KMPMRef> m_pmRefDownloadList;                    // 实际上需要安装的软件列表
    vector<KMPMRef> m_pmRefOfflineList;                     // 实际上需要安装的软件列表
    vector<KMPMRef> m_pmRefSearchList;                      // 查询的软件列表
    vector<string> m_pmFuzzyMatchRefs;                      // 从依赖文件中模糊匹配到的 ref 列表
    vector<KMRef> m_pmQueryVector;                          // 本地指定信息安装列表
    vector<KMRef> m_pmUninstallVector;                      // 卸载列表
    vector<string> m_pmOldAVersion;                         // 旧版本app
    vector<string> m_pmSpecialIdVersion;                    // 特殊版本app
    vector<string> m_pmReserveDepend;                       // 保留depend依赖
    std::map<string, KMPMRef> m_pmRefMap;                   // 存储所有ref的map表
    set<string> m_installedFiles;                           // 存储所有已安装的文件列表
    stack<string> m_pmRelatedRefs;                          // 先遍历运行时，再遍历app。因此先压入app,之后提取的时候顺序不会乱
    vector<string> m_pmDownloadRefs;                        // 先遍历运行时，再遍历app。因此先压入app,之后提取的时候顺序不会乱
    vector<string> m_pmDependRefs;                          // 先遍历运行时，再遍历app。因此先压入app,之后提取的时候顺序不会乱
    std::vector<KMfilterInfo> m_upgragableList;             // 可更新的列表
    vector<string> m_pmCanNotInstallVector;                 // 不可以安装的包名
    std::multimap<std::string, std::string> m_pmOkBase;     // 离线包所需的在线依赖
    vector<string> m_pmInstallId;                           // 安装软件包名

    std::map<string, KMRef> m_mapRefs; //ref映射表
    shared_ptr<KMOSTreeHandler> m_ostreeHandler;
    shared_ptr<KMOKFile> m_okFile;               // ok 文件解压
    shared_ptr<KMOABMetadata> m_okMetadata;      // ok 文件解压中的 metadata 数据
    shared_ptr<KMHostAppDeploy> m_hostAppDeploy; // hostapp deploy
};

KMPackageManager::KMPackageManager()
    : d(make_shared<Private>())
{
    initializeFixedModifications();
    d->m_folder = shared_ptr<KMFolder>(new KMFolder(KAIMING_SYSTEM_PATH, false));
    d->m_curl = shared_ptr<KMCurl>(new KMCurl());
    d->m_util = shared_ptr<KMUtil>(new KMUtil());
    d->m_env = shared_ptr<KMEnv>(new KMEnv());
    d->m_parseSummaryJson = std::make_shared<KMParseSummaryJson>();
    d->m_formatter = shared_ptr<KMFormatter>(new KMFormatter());

    //智能指针构造函数初始化
    d->m_ostreeHandler = std::make_shared<KMOSTreeHandler>(d->m_folder->getRepoPath()); //opt/kaiming/repo
    d->m_okFile = shared_ptr<KMOKFile>(new KMOKFile());                                 // ok 文件解压
    d->m_okMetadata = shared_ptr<KMOABMetadata>(new KMOABMetadata());                   // ok 文件解压中的 metadata 数据
    d->m_hostAppDeploy = std::make_shared<KMHostAppDeploy>(d->m_folder->getInfoPath());
    d->m_list = std::make_shared<KMList>();
}

KMPackageManager::~KMPackageManager()
{
}

int KMPackageManager::processTask(TaskType taskType, const vector<string> &vectorIds, PackageManagerOptions pmOptions)
{
    // 环境校验
    if (!d->m_env->passEnvCheck())
    {
        kmlogger.error("Failed to check environments!");
        return KM_ERROR_CHECK_ENV_FAILED;
    }

    d->m_folder->ensureRepoForSudo(false);

    // 根据任务类型执行不同的任务
    switch (taskType)
    {
    case TASK_TYPE_INSTALL:
        return install(vectorIds, pmOptions);

    case TASK_TYPE_UPDATE:
        return update(vectorIds, pmOptions);

    case TASK_TYPE_SEARCH:
        return search(vectorIds[0], pmOptions);

    case TASK_TYPE_UNINSTALL:
        return uninstall(vectorIds, pmOptions);

    case TASK_TYPE_REPAIR:
        return repair(vectorIds, pmOptions);

    case TASK_TYPE_UPGRADE:
        return install(vectorIds, pmOptions);
    default:
        break;
    }

    return KM_ERROR_NO;
}

int KMPackageManager::install(const vector<string> &vectorIds, PackageManagerOptions pmOptions)
{
    if (pmOptions.isPrint)
    {
        std::cout << _("Search matching packages ...") << std::endl;
    }

    d->m_pmIsForceInstall = pmOptions.force;
    int countOfflinePackages = 0;
    for (auto name : vectorIds)
    {
        kmlogger.info("Install app %s", name.c_str());
        bool isMatchedRefFlag = false;

        KMPMRef pmRef = d->m_util->parseKMPMRef(name);
        d->m_util->printKMPMRef(name, pmRef);

        // 如果用户指定了 arch，则覆盖默认的 arch
        if (!pmOptions.arch.empty())
        {
            pmRef.ref.arch = pmOptions.arch;
        }

        if (!pmOptions.kind.empty())
        {
            pmRef.ref.kind = pmOptions.kind;
        }

        if (pmRef.refSourceType == KMPACKAGE_SOURCE_TYPE_OK)
        {
            // 开始处理离线包
            ++countOfflinePackages;
        }
    }

    if (countOfflinePackages != 0 && countOfflinePackages != vectorIds.size())
    {
        cout << _("Please check the package is all online package or offline package!") << endl;
        return KM_ERROR_INSTALL_FAILED;
    }

    if (countOfflinePackages != 0)
    {
        // // 暂时先只安装一个离线包，之后优化
        // if (vectorIds.size() > 1)
        // {
        //     cout << _("Only support one offline package!") << endl;
        //     return KM_ERROR_INSTALL_FAILED;
        // }
        return installOfflinePackage(vectorIds, pmOptions);
    }

    // 下载和解析清单依赖文件 summary.json
    string ostreeRepoUrl = d->m_ostreeHandler->getRemoteUrl(REPO_NAME);
    fs::path summaryUrl = fs::path(ostreeRepoUrl) / "summary.json";

    int getSummaryState = getRemoteSummary(summaryUrl.string(), false);
    if (getSummaryState != KM_ERROR_NO)
    {
        kmlogger.error("Get and parse summary.json file error!");
        return getSummaryState;
    }

    // 遍历取得所有符合条件的依赖列表
    int getFuzzyMatchState = getAllFuzzyMatchRefVector(vectorIds, pmOptions, TASK_TYPE_INSTALL);
    if (getFuzzyMatchState != KM_ERROR_NO)
    {
        kmlogger.error("Get fuzzy match ref error!");
        return getFuzzyMatchState;
    }

    // 获取必要安装列表
    getUsefulRefList();
    // 获取真正安装列表
    getDownloadRefList(TASK_TYPE_INSTALL);

    if (d->m_pmRefDownloadList.empty())
    {
        return KM_ERROR_ALREADY_INSTALLED;
    }

    // 展示下载列表信息
    printDownloadRefList(d->m_pmRefDownloadList);

    // 用户选择是否展示信息
    if (!printAssumeYesChoice(pmOptions))
    {
        return KM_ERROR_NOT_INSTALLED;
    };

    // 检测当前网址是否可以访问
    if (KMStringUtils::contains(ostreeRepoUrl, "http"))
    {
        if (KMCurl::checkURL(ostreeRepoUrl, 3))
        {
            kmlogger.debug("URL is accessible!");
        }
        else
        {
            kmlogger.error("Failed to access URL!");
            return KM_ERROR_CURL_CHECK_URL_FAILED;
        }
    }

    // 下载包
    int pullFlag = false;
    pullFlag = d->m_ostreeHandler->pull(REPO_NAME, d->m_pmDownloadRefs, progressCallback);
    if (!pullFlag)
    {
        return KM_ERROR_PULL_FAILED;
    }
    for (auto &ref : d->m_pmOldAVersion)
    {
        if (!removeSpecificDeployFiles(ref))
        {
            return KM_ERROR_DELETE_DIRECTORY_FAILED;
        }
        if (!d->m_pmIsForceInstall)
        {
            if (!deleteSpecificRefsFromRepo(ref))
            {
                return KM_ERROR_DELETE_REF_TAG_FAILED;
            }
        }
    }

    if (!pmOptions.skipSignatureVerification)
    {
        int verifyResult = KM_ERROR_NO;
        for (auto &downloadRef : d->m_pmRefDownloadList)
        {
            fs::path tmpPath = fs::path("/tmp");

            pullFromOstree(tmpPath, downloadRef);

            fs::path idPath = tmpPath / downloadRef.ref.id;
            // 验签
            verifyResult = verifySignature(downloadRef.ref.id, downloadRef.ref.kind, idPath);

            if (!removeTmpDeployFiles(downloadRef.ref.id))
            {
                kmlogger.info("删除临时目录失败");
                std::cout << _("Failed to delete temporary directory") << std::endl;
                verifyResult = KM_ERROR_DELETE_DIRECTORY_FAILED;
            }
        }

        if (verifyResult != KM_ERROR_NO)
        {
            return verifyResult;
        }
        else
        {
            // 部署包
            return deploy();
        }
    }
    else
    {
        kmlogger.info("用户不进行验签，跳过验签步骤");
        // 部署包
        return deploy();
    }
}

int KMPackageManager::installOfflinePackage(const vector<string> &vectorIds, PackageManagerOptions pmOptions)
{
    vector<string> onlinePackageIds; // 离线包的 id 列表
    std::map<string, string> mountMap;
    bool printLog = false;
    bool canNotInstall = false;

    for (auto name : vectorIds)
    {
        kmlogger.debug("Install offline packages %s", name.c_str());
        bool isMatchedRefFlag = false;

        KMPMRef pmRef = d->m_util->parseKMPMRef(name);
        d->m_util->printKMPMRef(name, pmRef);

        if (d->m_pmIsThereDbusService)
        {
            const char *dbusmethod = "GetPackageInstallStatus";
            const char *idChar = pmRef.ref.id.c_str();
            const char *dbusInterface = "com.kylin.kysdk.applicationsec.antiinstall";
            int getDbusReturn = d->m_parseSummaryJson->getDbus(dbusInterface, dbusmethod, idChar);

            if (getDbusReturn == -1)
            {
                d->m_pmIsThereDbusService = false;
            }
            if (getDbusReturn == 1)
            {
                canNotInstall = true;
                std::cout << "can not install " << pmRef.ref.id << std::endl;
                continue;
            }
        }

        // 如果用户指定了 arch，则覆盖默认的 arch
        if (!pmOptions.arch.empty())
        {
            pmRef.ref.arch = pmOptions.arch;
        }

        if (!pmOptions.kind.empty())
        {
            pmRef.ref.kind = pmOptions.kind;
        }

        if (pmRef.refSourceType == KMPACKAGE_SOURCE_TYPE_OK)
        {
            // 开始处理离线包
            int retInstallOKFile = installFromOK(name, onlinePackageIds, pmOptions, mountMap, canNotInstall);

            if (retInstallOKFile == KM_ERROR_SIGN_FAILED)
            {
                canNotInstall = true;
                continue;
            }

            if (retInstallOKFile == KM_ERROR_ALREADY_INSTALLED)
            {
                if (!d->m_pmFuzzyMatchRefs.empty())
                {
                    d->m_pmFuzzyMatchRefs.pop_back();
                }
                continue;
            }

            if (retInstallOKFile == KM_ERROR_LOAD_FILE_FAILED && d->m_pmFuzzyMatchRefs.empty())
            {
                return KM_ERROR_LOAD_FILE_FAILED;
            }

            if (retInstallOKFile != KM_ERROR_NO && d->m_pmFuzzyMatchRefs.empty())
            {
                return retInstallOKFile;
            }
        }

        for (const auto &ref : d->m_pmFuzzyMatchRefs)
        {
            kmlogger.debug("fuzzy match ref : %s", ref.c_str());
        }

        // 依赖列表去重
        KMUtil::removeDuplicateValues(d->m_pmFuzzyMatchRefs);

        if (d->m_pmFuzzyMatchRefs.empty())
        {
            printf("%s\n", _("The package has been installed!"));
            d->m_okFile->umountOK();
            return KM_ERROR_ALREADY_INSTALLED;
        }
    }

    for (auto it = onlinePackageIds.begin(); it != onlinePackageIds.end();)
    {
        string onlinId = *it;
        KMPMRef idRef = d->m_util->parseKMPMRef(onlinId);
        bool isMatchedRefFlag = false;
        for (auto ref : d->m_pmFuzzyMatchRefs)
        {
            KMPMRef fuzzyRef = d->m_util->parseKMPMRef(ref);
            if (idRef.ref.id == fuzzyRef.ref.id && idRef.ref.kind == fuzzyRef.ref.kind && idRef.ref.arch == fuzzyRef.ref.arch && idRef.ref.module == fuzzyRef.ref.module && idRef.ref.channel == fuzzyRef.ref.channel)
            {
                if (idRef.ref.version[0] == '-')
                {
                    it = onlinePackageIds.erase(it);
                    isMatchedRefFlag = true;
                    break;
                }
                KMVersion idVersion = KMVersion::parse(idRef.ref.version, false);
                KMVersion fuzzyVersion = KMVersion::parse(fuzzyRef.ref.version, false);
                if (idVersion.getMajor() == fuzzyVersion.getMajor())
                {
                    it = onlinePackageIds.erase(it);
                    isMatchedRefFlag = true;
                    break;
                }
            }
        }
        if (!isMatchedRefFlag)
        {
            it++;
        }
    }

    string ostreeRepoUrl = d->m_ostreeHandler->getRemoteUrl(REPO_NAME);
    fs::path summaryUrl = fs::path(ostreeRepoUrl) / "summary.json";

    if (!onlinePackageIds.empty())
    {
        d->m_pmFuzzyMatchRefs.clear();
        int getSummaryState = getRemoteSummary(summaryUrl.string(), false);
        if (getSummaryState != KM_ERROR_NO)
        {
            kmlogger.error("Get and parse summary.json file error!");
            return getSummaryState;
        }

        int getFuzzyMatchState = getAllFuzzyMatchRefVector(onlinePackageIds, pmOptions, TASK_TYPE_INSTALL);

        if (!d->m_pmCanNotInstallVector.empty())
        {
            std::cout << _("The dependencies of these software packages cannot be satisfied: ") << std::endl;
        }

        for (auto notInstall : d->m_pmCanNotInstallVector)
        {
            std::unordered_set<std::string> keysToDelete;
            // 找出所有依赖 notInstall 的键
            for (auto it = d->m_pmOkBase.begin(); it != d->m_pmOkBase.end(); ++it)
            {
                if (it->second == notInstall)
                {
                    keysToDelete.insert(it->first);
                }
            }

            // 从 m_pmOkBase 中删除所有相关的键值对
            for (const auto &key : keysToDelete)
            {
                auto range = d->m_pmOkBase.equal_range(key);
                for (auto it = range.first; it != range.second; ++it)
                {
                    std::regex pattern(R"(([^/]+)/([^/]+)/([^/]+)/([^/]+)/([^/]+))");

                    for (auto itRef = d->m_pmFuzzyMatchRefs.begin(); itRef != d->m_pmFuzzyMatchRefs.end();)
                    {
                        KMPMRef pmFuzzyRef = KMUtil::parseSummaryFuzzyKMPMRef(*itRef);
                        std::smatch match;

                        if (std::regex_search(it->second, match, pattern))
                        {
                            std::string id = match[1].str();
                            std::string version = match[2].str();
                            std::string module = match[3].str();
                            std::string channel = match[4].str();
                            std::string arch = match[5].str();

                            if (pmFuzzyRef.ref.id == id && pmFuzzyRef.ref.module == module && pmFuzzyRef.ref.channel == channel && pmFuzzyRef.ref.arch == arch)
                            {
                                if (version == "-" || (pmFuzzyRef.ref.version[0] == version[0]))
                                {
                                    itRef = d->m_pmFuzzyMatchRefs.erase(itRef);
                                    continue;
                                }
                            }
                        }
                        ++itRef;
                    }
                }
                std::cout << key << std::endl;
                d->m_pmOkBase.erase(key);
                printLog = true;
                // 从 m_pmRefOfflineList 中删除相关的引用
                auto newEnd = std::remove_if(d->m_pmRefOfflineList.begin(), d->m_pmRefOfflineList.end(), [this, key](const KMPMRef &element)
                                             { return element.ref.id == key; });
                d->m_pmRefOfflineList.erase(newEnd, d->m_pmRefOfflineList.end());
            }
        }

        if (getFuzzyMatchState != KM_ERROR_NO)
        {
            d->m_okFile->umountOK();
            kmlogger.error("Get fuzzy match ref error!");
            return getFuzzyMatchState;
        }

        // 获取必要安装列表
        getUsefulRefList();
        // 获取真正安装列表
        getDownloadRefList(TASK_TYPE_INSTALL);
    }

    if (d->m_pmRefOfflineList.empty() && d->m_pmRefDownloadList.empty())
    {
        if (printLog || canNotInstall)
        {
            d->m_okFile->umountOK();
            return KM_ERROR_NOT_INSTALLED;
        }
        else
        {
            printf("%s\n", _("This app has been installed!"));
            d->m_okFile->umountOK();
            return KM_ERROR_ALREADY_INSTALLED;
        }
    }
    else if (d->m_pmRefOfflineList.empty() && !d->m_pmRefDownloadList.empty())
    {
        // 展示下载列表信息
        printDownloadRefList(d->m_pmRefDownloadList);

        // 用户选择是否展示信息
        if (!printAssumeYesChoice(pmOptions))
        {
            d->m_okFile->umountOK();
            return KM_ERROR_NOT_INSTALLED;
        }

        // 检测当前网址是否可以访问
        if (KMStringUtils::contains(ostreeRepoUrl, "http"))
        {
            if (KMCurl::checkURL(ostreeRepoUrl, 3))
            {
                kmlogger.debug("URL is accessible!");
            }
            else
            {
                kmlogger.error("Failed to access URL!");
                return KM_ERROR_CURL_CHECK_URL_FAILED;
            }
        }

        // 下载包
        int pullFlag = false;
        pullFlag = d->m_ostreeHandler->pull(REPO_NAME, d->m_pmDownloadRefs, progressCallback);
        if (!pullFlag)
        {
            d->m_okFile->umountOK();
            return KM_ERROR_PULL_FAILED;
        }

        // 部署在线下载包
        kmlogger.info("Only deploy online package!");

        for (auto &ref : d->m_pmOldAVersion)
        {
            if (!removeSpecificDeployFiles(ref))
            {
                return KM_ERROR_DELETE_DIRECTORY_FAILED;
            }
            if (!d->m_pmIsForceInstall)
            {
                if (!deleteSpecificRefsFromRepo(ref))
                {
                    return KM_ERROR_DELETE_REF_TAG_FAILED;
                }
            }
        }

        return deploy();
    }
    else if (!d->m_pmRefOfflineList.empty() && d->m_pmRefDownloadList.empty())
    {
        // 展示下载列表信息
        printDownloadRefList(d->m_pmRefOfflineList);

        // 用户选择是否展示信息
        if (!printAssumeYesChoice(pmOptions))
        {
            d->m_okFile->umountOK();
            return KM_ERROR_NOT_INSTALLED;
        }

        // 部署离线包
        kmlogger.info("Only deploy offline package!");

        for (auto &ref : d->m_pmOldAVersion)
        {
            if (!removeSpecificDeployFiles(ref))
            {
                return KM_ERROR_DELETE_DIRECTORY_FAILED;
            }
            if (!d->m_pmIsForceInstall)
            {
                if (!deleteSpecificRefsFromRepo(ref))
                {
                    return KM_ERROR_DELETE_REF_TAG_FAILED;
                }
            }
        }

        return deployOfflinePackage(mountMap);
    }
    else
    {
        // 展示下载列表信息
        vector<KMPMRef> allRefList;
        allRefList.insert(allRefList.end(), d->m_pmRefOfflineList.begin(), d->m_pmRefOfflineList.end());
        allRefList.insert(allRefList.end(), d->m_pmRefDownloadList.begin(), d->m_pmRefDownloadList.end());
        printDownloadRefList(allRefList);

        for (auto &offline : d->m_pmRefOfflineList)
        {
            kmlogger.info("offline ref: %s\n", offline.completedRef.c_str());
        }

        for (auto &online : d->m_pmRefDownloadList)
        {
            kmlogger.info("online ref: %s\n", online.completedRef.c_str());
        }

        // 用户选择是否展示信息
        if (!printAssumeYesChoice(pmOptions))
        {
            d->m_okFile->umountOK();
            return KM_ERROR_NOT_INSTALLED;
        }

        // 部署离线包
        kmlogger.info("Deploy offline package!");

        // 检测当前网址是否可以访问
        if (KMStringUtils::contains(ostreeRepoUrl, "http"))
        {
            if (KMCurl::checkURL(ostreeRepoUrl, 3))
            {
                kmlogger.debug("URL is accessible!");
            }
            else
            {
                kmlogger.error("Failed to access URL!");
                return KM_ERROR_CURL_CHECK_URL_FAILED;
            }
        }

        // 下载包
        int pullFlag = false;
        pullFlag = d->m_ostreeHandler->pull(REPO_NAME, d->m_pmDownloadRefs, progressCallback);
        if (!pullFlag)
        {
            return KM_ERROR_PULL_FAILED;
        }

        // 部署在线下载包
        kmlogger.info("Deploy online package!");
        int retDeploy = deploy();
        if (retDeploy != KM_ERROR_NO)
        {
            return retDeploy;
        }

        for (auto &ref : d->m_pmOldAVersion)
        {
            if (!removeSpecificDeployFiles(ref))
            {
                return KM_ERROR_DELETE_DIRECTORY_FAILED;
            }
            if (!d->m_pmIsForceInstall)
            {
                if (!deleteSpecificRefsFromRepo(ref))
                {
                    return KM_ERROR_DELETE_REF_TAG_FAILED;
                }
            }
        }
        return deployOfflinePackage(mountMap);
    }
}

int KMPackageManager::installSoftwareCenter(string name, string version, int &error, KMPMProgressCallBack installCallback)
{
    kmlogger.debug("begin install");
    globalProgressCallback = installCallback;

    int ret = KM_ERROR_NOTHING_TO_DO;

    PackageManagerOptions pmOption;
    vector<string> vecInstall;

    pmOption.assumeYes = true;
    pmOption.skipSignatureVerification = true;

    vecInstall.push_back(name);

    ret = install(vecInstall, pmOption);

    error = ret;

    kmlogger.debug("end install");
    return ret;
}

int KMPackageManager::updateSoftwareCenter(string name, string version, int &error, KMPMProgressCallBack updateCallback)
{
    kmlogger.debug("begin update");
    globalProgressCallback = updateCallback;

    int ret = KM_ERROR_NOTHING_TO_DO;

    PackageManagerOptions pmOption;
    vector<string> vecUpdate;

    pmOption.assumeYes = true;

    vecUpdate.push_back(name);

    ret = update(vecUpdate, pmOption);

    error = ret;

    kmlogger.debug("end update");
    return ret;
}

int KMPackageManager::uninstallSoftwareCenter(string name, int &error)
{
    kmlogger.debug("begin uninstall");

    int ret = KM_ERROR_NOTHING_TO_DO;

    PackageManagerOptions pmOption;
    vector<string> vecUninstall;

    pmOption.assumeYes = true;

    vecUninstall.push_back(name);

    ret = uninstall(vecUninstall, pmOption);

    error = ret;

    kmlogger.debug("end uninstall");
    return ret;
}

int KMPackageManager::installFromOK(const string &name, vector<string> &onlinePackageIds, PackageManagerOptions pmOptions, std::map<string, string> &mountMap, bool &canNotInstall)
{
    d->m_okMetadata->layers.clear();
    size_t pos = name.find(".ok");
    string mountPoint = "/tmp/" + name.substr(0, pos);

    d->m_okFile->setMountPoint(mountPoint);
    // d->m_okFile->setOKFilePath(name);

    if (!d->m_okFile->loadOKFile(name))
    {
        std::cout << "Failed to load file: " << name << std::endl;
        return KM_ERROR_LOAD_FILE_FAILED;
    }

    d->m_okFile->mountOK(mountPoint);
    d->m_okMetadata = d->m_okFile->getMetadata();
    std::vector<KMInfoJson> layers = d->m_okMetadata->layers;

    if (!pmOptions.skipSignatureVerification)
    {
        std::string filename = name.substr(0, pos);
        size_t lastSlashPos = filename.find_last_of("/\\");

        filename = (lastSlashPos != std::string::npos) ? filename.substr(lastSlashPos + 1) : filename;

        std::string okId, okKind, okModule;
        if (layers.size() == 1)
        {
            okId = layers[0].id;
            okKind = layers[0].kind;
            okModule = layers[0].module;
        }
        std::string path = mountPoint + "/" + "layers/" + okId + "/" + okModule;
        fs::path modulePath = path;

        int errorCode = verifySignature(okId, okKind, modulePath);

        if (errorCode != KM_ERROR_NO)
        {
            return KM_ERROR_SIGN_FAILED;
        }
    }

    if (layers.empty())
    {
        kmlogger.error("No layers found in OK file!");
        return KM_ERROR_NO_LAYERS_FOUND;
    }

    if (layers.size() == 1)
    {
        for (auto layer : layers)
        {
            KMPMRef pmRef;
            string base = layer.base;
            string runtime = layer.runtime;
            vector<string> depends = layer.annotations.depends;
            pmRef.ref.id = layer.id;
            pmRef.ref.channel = layer.channel;
            pmRef.ref.version = layer.version;
            pmRef.ref.module = layer.module;
            pmRef.ref.kind = layer.kind;
            pmRef.installedSize = layer.size;
            pmRef.name = layer.name;
            pmRef.description = layer.description;
            mountMap[layer.id + layer.module] = mountPoint;

            if (d->m_pmIsThereDbusService)
            {
                const char *dbusmethod = "GetPackageInstallStatus";
                const char *idChar = pmRef.ref.id.c_str();
                const char *dbusInterface = "com.kylin.kysdk.applicationsec.antiinstall";
                int getDbusReturn = d->m_parseSummaryJson->getDbus(dbusInterface, dbusmethod, idChar);

                if (getDbusReturn == -1)
                {
                    d->m_pmIsThereDbusService = false;
                }
                if (getDbusReturn == 1)
                {
                    canNotInstall = true;
                    std::cout << "can not uninstall " << pmRef.ref.id << std::endl;
                    return KM_ERROR_INSTALL_FAILED;
                }
            }

            for (auto arch : layer.archs)
            {
                if (arch == KMUtil::kmGetArch())
                {
                    pmRef.ref.arch = arch;
                }
            }

            pmRef.completedRef = KMUtil::dupKMPMCompletedRef(pmRef);
            d->m_pmRefMap[layer.id] = pmRef;

            if (pmRef.ref.arch.empty())
            {
                std::cout << "Not support this arch" << std::endl;
                return KM_ERROR_NOT_SUPPORT_ARCH;
            }

            d->m_pmSpecialIdVersion.push_back(pmRef.ref.id + pmRef.ref.version);

            if (currentFuzzyRefIsInstalled(pmRef))
            {
                //不强制安装已经安装的应用
                if (!pmOptions.force)
                {
                    std::cout << _("This package has been installed! ") << pmRef.ref.id << std::endl;
                    return KM_ERROR_FORCE_INSTALL;
                }

                std::string removeStr;

                if (pmRef.ref.module == "binary")
                {
                    fs::path rootPath = d->m_folder->getLayersPath();
                    removeStr = rootPath.string() + "/" + pmRef.completedRef;
                }

                if (pmRef.ref.module == "devel")
                {
                    fs::path develDir = fs::path(KMStorageDir::getUserBaseDirLocation());
                    removeStr = develDir.string() + "/" + pmRef.completedRef;
                }
                removeSpecificDeployFiles(removeStr);
            }

            std::string completedRef = pmRef.ref.id + "/" + pmRef.ref.version + "/" + pmRef.ref.module + "/" + pmRef.ref.channel + "/" + pmRef.ref.arch;
            d->m_pmFuzzyMatchRefs.push_back(completedRef);
            d->m_pmRefOfflineList.push_back(pmRef);

            if (pmRef.ref.kind != "base")
            {
                if (!runtime.empty())
                {
                    KMPMRef pmRuntimeRef = KMUtil::parseSummaryKMPMRef(runtime);
                    pmRuntimeRef.ref.module = layer.module;
                    std::string runtimeStr = pmRuntimeRef.ref.id + "/" + pmRuntimeRef.ref.version + "/" + layer.module + "/" + layer.channel + "/" + pmRef.ref.arch;

                    if (!d->m_parseSummaryJson->currentFuzzyRefIsInstalled(pmRuntimeRef))
                    {
                        d->m_pmOkBase.insert({pmRef.ref.id, runtimeStr});
                        onlinePackageIds.push_back(runtimeStr);
                    }
                }

                if (!base.empty())
                {
                    KMPMRef pmBaseRef = KMUtil::parseSummaryKMPMRef(base);
                    pmBaseRef.ref.module = layer.module;
                    std::string baseStr = pmBaseRef.ref.id + "/" + pmBaseRef.ref.version + "/" + layer.module + "/" + layer.channel + "/" + pmRef.ref.arch;
                    if (!d->m_parseSummaryJson->currentFuzzyRefIsInstalled(pmBaseRef))
                    {
                        d->m_pmOkBase.insert({pmRef.ref.id, baseStr});
                        onlinePackageIds.push_back(baseStr);
                    }
                }

                if (!depends.empty())
                {
                    for (std::string dependStr : depends)
                    {
                        KMPMRef pmDependRef;
                        std::vector<std::string> tokens;
                        std::stringstream ss(dependStr);
                        std::string token;
                        while (std::getline(ss, token, '/'))
                        {
                            tokens.push_back(token); // 将切分后的部分添加到 tokens 向量中
                        }

                        if (tokens.empty())
                        {
                            kmlogger.debug("Empty dependency string found: : %s", dependStr.c_str());
                            continue;
                        }

                        if (dependStr.find('/') != std::string::npos)
                        {
                            pmDependRef.ref.arch = pmRef.ref.arch;
                            pmDependRef.ref.id = tokens[0];
                            pmDependRef.ref.version = tokens.size() > 1 ? tokens[1] : "";
                            pmDependRef.ref.module = layer.module;
                            pmDependRef.ref.channel = layer.channel;

                            if (!currentFuzzyRefIsInstalled(pmDependRef))
                            {
                                dependStr = dependStr + "/" + layer.module + "/" + layer.channel + "/" + pmRef.ref.arch;
                                d->m_pmOkBase.insert({pmRef.ref.id, dependStr});
                                onlinePackageIds.push_back(dependStr);
                            }
                        }
                        else
                        {
                            pmDependRef.ref.arch = pmRef.ref.arch;
                            pmDependRef.ref.id = tokens[0];
                            pmDependRef.ref.version = "";
                            pmDependRef.ref.module = layer.module;
                            pmDependRef.ref.channel = layer.channel;
                            if (!currentRefIsUninstalled(pmDependRef))
                            {
                                dependStr = dependStr + "/" + "-" + "/" + layer.module + "/" + layer.channel + "/" + pmRef.ref.arch;
                                d->m_pmOkBase.insert({pmRef.ref.id, dependStr});
                                onlinePackageIds.push_back(dependStr);
                            }
                        }
                    }
                }
            }
        }
    }
    else
    {
        for (auto layer : layers)
        {
            KMPMRef pmRef;
            string base = layer.base;
            string runtime = layer.runtime;

            pmRef.ref.id = layer.id;
            pmRef.ref.channel = layer.channel;
            pmRef.ref.version = layer.version;
            pmRef.ref.module = layer.module;
            pmRef.ref.kind = layer.kind;
            pmRef.installedSize = layer.size;
            pmRef.name = layer.name;
            pmRef.description = layer.description;

            for (auto arch : layer.archs)
            {
                if (arch == KMUtil::kmGetArch())
                {
                    pmRef.ref.arch = arch;
                }
            }

            if (d->m_pmIsThereDbusService)
            {
                const char *dbusmethod = "GetPackageInstallStatus";
                const char *idChar = pmRef.ref.id.c_str();
                const char *dbusInterface = "com.kylin.kysdk.applicationsec.antiinstall";
                int getDbusReturn = d->m_parseSummaryJson->getDbus(dbusInterface, dbusmethod, idChar);

                if (getDbusReturn == -1)
                {
                    d->m_pmIsThereDbusService = false;
                }
                if (getDbusReturn == 1)
                {
                    canNotInstall = true;
                    std::cout << "can not uninstall " << pmRef.ref.id << std::endl;
                    return KM_ERROR_INSTALL_FAILED;
                }
            }

            pmRef.completedRef = KMUtil::dupKMPMCompletedRef(pmRef);

            if (pmRef.ref.arch.empty())
            {
                std::cout << "Not support this arch" << std::endl;
                return KM_ERROR_NOT_SUPPORT_ARCH;
            }

            if (currentRefIsInstalled(pmRef))
            {
                //不强制安装已经安装的应用
                if (!pmOptions.force)
                {
                    return KM_ERROR_FORCE_INSTALL;
                }
            }
            d->m_pmRefOfflineList.push_back(pmRef);
        }
    }

    return KM_ERROR_NO;
}

int KMPackageManager::uninstall(const vector<string> &vectorIds, PackageManagerOptions pmOptions)
{
    std::vector<std::string> perUninstallList; // 待删除列表
    bool isUninstallable = true;
    vector<std::string> UninstallVector;
    vector<std::string> dependVector;

    // 遍历每个待卸载的应用ID
    for (auto name : vectorIds)
    {
        kmlogger.debug("Uninstall app %s", name.c_str());
        bool isMatchedRefFlag = false;

        KMPMRef pmRef = d->m_util->parseKMPMRef(name);
        d->m_util->printKMPMRef(name, pmRef);
        pmRef.ref.arch = pmOptions.arch;

        if (!currentRefIsUninstalled(pmRef))
        {
            kmlogger.pinfo("%s not installed!", name.c_str());
            continue;
        }
        kmlogger.pinfo("find installed package %s", name.c_str());

        printUninstallList();

        // 用户选择是否展示信息
        if (!printAssumeUninstallYesChoice(pmOptions))
        {
            return KM_ERROR_NOT_UNINSTALLED;
        }

        bool needDevelMap = false;
        bool needRootMap = false;
        for (auto it : d->m_pmUninstallVector)
        {
            if (needDevelMap && needRootMap)
            {
                break;
            }
            if (it.module == "binary")
            {
                fs::path rootPath = d->m_folder->getLayersPath();
                std::unordered_map<std::string, KMInfoJson> rootMap = loadAllInfoJsonFiles(rootPath);
                m_infoMap.insert(rootMap.begin(), rootMap.end());
                needRootMap = true;
            }
            else
            {
                fs::path develDir = fs::path(KMStorageDir::getUserBaseDirLocation());
                std::unordered_map<std::string, KMInfoJson> develMap = loadAllInfoJsonFiles(develDir);
                m_infoMap.insert(develMap.begin(), develMap.end());
                needDevelMap = true;
            }
        }

        if (d->m_pmUninstallVector[0].kind == "app")
        {
            if (d->m_pmIsThereDbusService)
            {
                const char *dbusmethod = "GetPackageUninstallStatus";
                const char *idChar = d->m_pmUninstallVector[0].id.c_str();
                const char *dbusInterface = "com.kylin.kysdk.applicationsec.antiuninstall";
                int getDbusReturn = d->m_parseSummaryJson->getDbus(dbusInterface, dbusmethod, idChar);

                if (getDbusReturn == -1)
                {
                    d->m_pmIsThereDbusService = false;
                }
                if (getDbusReturn == 1)
                {
                    isUninstallable = false;
                    kmlogger.error("Can not uninstall %s", idChar);
                    std::cout << "Can not uninstall " << d->m_pmUninstallVector[0].id << std::endl;
                    return isUninstallable;
                }
            }

            if (pmOptions.deletedataYes)
            {
                deleteUserData(name);
            }

            deleteHostAppFiles(); // deleteHostAppFiles before removeDeployFiles

            KMRef appRemoveRef =  d->m_pmUninstallVector[0];

            string removeStr = appRemoveRef .baseInstallDir + "/" + appRemoveRef.channel + "/" + appRemoveRef.arch + "/" + appRemoveRef.kind + "/" + appRemoveRef.id + "/" + appRemoveRef.module + "/" + appRemoveRef.version;
            if (!removeSpecificDeployFiles(removeStr))
            {
                return KM_ERROR_UNINSTALL_FAILED;
            }

            // if (pmOptions.unusedYes)
            // {
            //     // 查找并处理该应用的依赖项
            //     findDependencies(name, perUninstallList);

            // } //默认不删除runtime和base
        }

        vector<KMRef> unRemove = reserveVersion();

        if (d->m_pmUninstallVector[0].kind == "depend")
        {
            if (!unRemove.empty())
            {
                d->m_pmReserveDepend.push_back(unRemove[0].id);
            }

            for (auto &uninstallRef : d->m_pmUninstallVector)
            {
                if (removeDepend(pmRef.ref.id, uninstallRef, UninstallVector))
                {
                    std::string firstDepend = UninstallVector[0];

                    if (!UninstallVector.empty())
                    {
                        UninstallVector.erase(UninstallVector.begin());
                    }

                    printUninstallDependsOnList(UninstallVector);

                    if (UninstallVector.empty() || isContinueUninstall())
                    {
                        UninstallVector.push_back(firstDepend);

                        for (auto &uninstallStr : UninstallVector)
                        {
                            removeHostAppDeployFiles(uninstallStr); // remove hostapp first, need to read info.json
                            removeSpecificDeployFiles(uninstallStr);
                            deleteSpecificRefsFromRepo(uninstallStr);
                        }
                    }
                }
                else
                {
                    std::cout << "can not uninstall " << d->m_pmUninstallVector[0].id << std::endl;
                    break;
                }
            }
        }

        if (d->m_pmUninstallVector[0].kind == "runtime" || d->m_pmUninstallVector[0].kind == "base")
        {

            for (auto it = d->m_pmUninstallVector.begin(); it != d->m_pmUninstallVector.end(); it++)
            {
                auto &uninstallIt = *it;
                for (auto &queryIt : unRemove)
                {
                    if (uninstallIt.arch == queryIt.arch && uninstallIt.channel == queryIt.channel && uninstallIt.module == queryIt.module)
                    {
                        if (uninstallIt.version != queryIt.version)
                        {
                            KMVersion uninstallVersion = KMVersion::parse(uninstallIt.version, false);
                            KMVersion queryVersion = KMVersion::parse(queryIt.version, false);
                            if (uninstallVersion.getMajor() == queryVersion.getMajor())
                            {
                                // directUninstallVector.push_back(uninstallIt);
                                std::string removeUninstallStr = uninstallIt.baseInstallDir + "/" + uninstallIt.channel + "/" + uninstallIt.arch + "/" + uninstallIt.kind + "/" + uninstallIt.id + "/" + uninstallIt.module + "/" + uninstallIt.version;
                                removeSpecificDeployFiles(removeUninstallStr);
                                deleteSpecificRefsFromRepo(removeUninstallStr);
                                it = d->m_pmUninstallVector.erase(it);
                                it--;
                                break;
                            }
                        }
                    }
                }
            }

            if (!d->m_pmUninstallVector.empty())
            {
                for (auto it = d->m_pmUninstallVector.begin(); it != d->m_pmUninstallVector.end(); it++)
                {
                    auto &uninstallIt = *it;

                    if (d->m_pmIsThereDbusService)
                    {
                        const char *dbusmethod = "GetPackageUninstallStatus";
                        const char *idChar = uninstallIt.id.c_str();
                        const char *dbusInterface = "com.kylin.kysdk.applicationsec.antiuninstall";
                        int getDbusReturn = d->m_parseSummaryJson->getDbus(dbusInterface, dbusmethod, idChar);

                        if (getDbusReturn == -1)
                        {
                            d->m_pmIsThereDbusService = false;
                        }
                        if (getDbusReturn == 1)
                        {
                            isUninstallable = false;
                            kmlogger.error("Can not uninstall %s", idChar);
                            std::cout << "can not uninstall " << uninstallIt.id << std::endl;
                            break;
                        }
                    }

                    std::string removeUninstallStr = uninstallIt.baseInstallDir + "/" + uninstallIt.channel + "/" + uninstallIt.arch + "/" + uninstallIt.kind + "/" + uninstallIt.id + "/" + uninstallIt.module + "/" + uninstallIt.version;
                    std::string findStr = uninstallIt.channel + ":" + uninstallIt.id + "/" + uninstallIt.version[0];
                    vector<std::string> willUninstallVector;

                    for (auto it = m_infoMap.begin(); it != m_infoMap.end();)
                    {
                        auto pair = *it;
                        KMInfoJson &info = pair.second;

                        if (info[uninstallIt.kind].find(findStr) != std::string::npos && info[uninstallIt.kind].find(uninstallIt.arch) != std::string::npos && info.module == uninstallIt.module)
                        {
                            if (d->m_pmIsThereDbusService)
                            {
                                const char *dbusmethod = "GetPackageUninstallStatus";
                                const char *idChar = info.id.c_str();
                                const char *dbusInterface = "com.kylin.kysdk.applicationsec.antiuninstall";
                                int getDbusReturn = d->m_parseSummaryJson->getDbus(dbusInterface, dbusmethod, idChar);

                                if (getDbusReturn == -1)
                                {
                                    d->m_pmIsThereDbusService = false;
                                }

                                if (getDbusReturn == 1)
                                {
                                    isUninstallable = false;
                                    kmlogger.error("Can not uninstall %s", idChar);
                                    std::cout << info.id << " depends on " << name << ",but " << info.id << " can not be uninstalled" << std::endl;
                                    break;
                                }
                            }

                            if (info.kind == "runtime")
                            {
                                std::string runtimeStr = info.channel + ":" + info.id + "/" + info.version[0];

                                for (auto secondIt = m_infoMap.begin(); secondIt != m_infoMap.end();)
                                {
                                    auto secondPair = *secondIt;
                                    KMInfoJson &secondInfo = secondPair.second;

                                    if (secondInfo[info.kind].find(runtimeStr) != std::string::npos && secondInfo[info.kind].find(uninstallIt.arch) != std::string::npos && secondInfo.module == uninstallIt.module)
                                    {
                                        if (d->m_pmIsThereDbusService)
                                        {
                                            const char *dbusmethod = "GetPackageUninstallStatus";
                                            const char *idChar = secondInfo.id.c_str();
                                            const char *dbusInterface = "com.kylin.kysdk.applicationsec.antiuninstall";
                                            int getDbusReturn = d->m_parseSummaryJson->getDbus(dbusInterface, dbusmethod, idChar);

                                            if (getDbusReturn == -1)
                                            {
                                                d->m_pmIsThereDbusService = false;
                                            }

                                            if (getDbusReturn == 1)
                                            {
                                                isUninstallable = false;
                                                kmlogger.error("Can not uninstall %s", idChar);
                                                std::cout << d->m_pmUninstallVector[0].id << " depends on " << name << ",but " << d->m_pmUninstallVector[0].id << " can not be uninstalled" << std::endl;
                                                break;
                                            }
                                        }

                                        std::string removeSecondStr = secondPair.first;
                                        willUninstallVector.push_back(removeSecondStr);
                                        secondIt = m_infoMap.erase(secondIt);
                                    }
                                    else
                                    {
                                        secondIt++;
                                    }
                                }

                                if (!isUninstallable)
                                {
                                    break;
                                }
                            }
                            if (info.kind == "depend")
                            {
                                KMRef dependRef;
                                dependRef.id = info.id;
                                dependRef.kind = info.kind;
                                dependRef.module = info.module;
                                dependRef.arch = d->m_pmUninstallVector[0].arch;
                                dependRef.version = info.version;
                                dependRef.baseInstallDir = fs::path(KMStorageDir::getUserBaseDirLocation());
                                isUninstallable = removeDepend(pmRef.ref.id, dependRef, willUninstallVector);

                                if (!isUninstallable)
                                {
                                    break;
                                }
                            }
                            std::string removeStr = pair.first;
                            willUninstallVector.push_back(removeStr);
                            it = m_infoMap.erase(it);
                        }
                        else
                        {
                            it++;
                        }
                    }

                    if (!isUninstallable)
                    {
                        break;
                    }

                    UninstallVector.push_back(removeUninstallStr);
                    dependVector.insert(dependVector.end(), willUninstallVector.begin(), willUninstallVector.end());
                    UninstallVector.insert(UninstallVector.end(), willUninstallVector.begin(), willUninstallVector.end());
                }

                printUninstallDependsOnList(dependVector);

                if (dependVector.empty() || isContinueUninstall())
                {
                    for (auto &uninstallStr : UninstallVector)
                    {
                        removeHostAppDeployFiles(uninstallStr);
                        removeSpecificDeployFiles(uninstallStr);
                        deleteSpecificRefsFromRepo(uninstallStr);
                    }
                }
            }
        }

    }
    return KM_ERROR_NO;
}

bool KMPackageManager::removeDepend(const string name, const KMRef uninstallRef, vector<std::string> &UninstallVector)
{
    bool isUninstallable = true;
    if (d->m_pmIsThereDbusService)
    {
        const char *dbusmethod = "GetPackageUninstallStatus";
        const char *idChar = uninstallRef.id.c_str();
        const char *dbusInterface = "com.kylin.kysdk.applicationsec.antiuninstall";
        int getDbusReturn = d->m_parseSummaryJson->getDbus(dbusInterface, dbusmethod, idChar);

        if (getDbusReturn == -1)
        {
            d->m_pmIsThereDbusService = false;
        }
        if (getDbusReturn == 1)
        {
            isUninstallable = false;
            if (uninstallRef.id == name)
            {
                kmlogger.error("Can not uninstall %s", idChar);
                std::cout << "can not uninstall " << name << std::endl;
                return isUninstallable;
            }
            else
            {
                kmlogger.error("Can not uninstall %s", idChar);
                std::cout << uninstallRef.id << " depends on " << name << ", but " << uninstallRef.id << " can not be uninstalled" << std::endl;
            }
        }
    }

    std::string removeUninstallStr = uninstallRef.baseInstallDir + "/" + uninstallRef.channel + "/" + uninstallRef.arch + "/" + uninstallRef.kind + "/" + uninstallRef.id + "/" + uninstallRef.module + "/" + uninstallRef.version;
    auto it = std::find(UninstallVector.begin(), UninstallVector.end(), removeUninstallStr);

    if (it != UninstallVector.end())
    {
        return true;
    }
    else
    {
        UninstallVector.push_back(removeUninstallStr);
    }

    for (auto it = m_infoMap.begin(); it != m_infoMap.end();)
    {
        auto pair = *it;
        KMInfoJson &info = pair.second;
        std::string uninstallArch;
        auto uninstallArchIt = std::find(info.archs.begin(), info.archs.end(), uninstallRef.arch);

        if (uninstallArchIt != info.archs.end())
        {
            uninstallArch = uninstallRef.arch;
        }
        else
        {
            uninstallArch = info.archs[0];
        }

        if (uninstallArch != uninstallRef.arch || info.module != uninstallRef.module || info.channel != uninstallRef.channel)
        {
            it++;
            continue;
        }

        for (auto dependstr : info.annotations.depends)
        {
            if (dependstr.find(uninstallRef.id) != std::string::npos)
            {

                KMVersion uninstallVersion = KMVersion::parse(uninstallRef.version, false);
                uint64_t uninstallVersionStr = uninstallVersion.getMajor();
                std::string dependInfo = uninstallRef.id + "/" + std::to_string(uninstallVersionStr);
                KMRef dependRef;
                dependRef.id = info.id;
                dependRef.kind = info.kind;
                dependRef.module = info.module;
                dependRef.channel = info.channel;
                dependRef.arch = uninstallArch;
                dependRef.version = info.version;
                dependRef.baseInstallDir = fs::path(KMStorageDir::getUserBaseDirLocation());
                if (dependstr.find(dependInfo) != std::string::npos)
                {
                    if (!removeDepend(name, dependRef, UninstallVector))
                    {
                        isUninstallable = false;
                        return isUninstallable;
                    }
                }
                else
                {
                    if (!d->m_pmReserveDepend.empty())
                    {
                        for (auto reserver : d->m_pmReserveDepend)
                        {
                            if (reserver == uninstallRef.id)
                            {
                                d->m_pmReserveDepend.push_back(info.id);
                            }
                            else
                            {
                                if (!removeDepend(name, dependRef, UninstallVector))
                                {
                                    isUninstallable = false;
                                    return isUninstallable;
                                }
                            }
                        }
                    }
                    else
                    {
                        if (!removeDepend(name, dependRef, UninstallVector))
                        {
                            isUninstallable = false;
                            return isUninstallable;
                        }
                    }
                }
            }
        }
        it++;
    }

    return isUninstallable;
}

bool KMPackageManager::isContinueUninstall()
{
    bool isContinueUninstall = false;
    while (true)
    {
        std::cout << _("The apps, which depends on those packages, will be uninstalled, do you want to continue (Y or N)?");
        std::string input;
        std::getline(std::cin, input);

        // 检查是否输入 'a'
        if (input == "y" || input == "Y")
        {
            isContinueUninstall = true;
            return isContinueUninstall;
        }
        else if (input == "n" || input == "N")
        {
            return isContinueUninstall;
        }
        else
        {
            continue;
        }
    }
}

void KMPackageManager::printUninstallList()
{
    int i = 1;
    printf("\n%-10s %-30s %-15s %-15s %-15s %-15s\n", _(" "), _("ID"), _("Channel"), _("Arch"), _("Module"), _("Version"));
    for (const auto &pmRef : d->m_pmQueryVector)
    {
        printf("%-10d %-30s %-15s %-15s %-15s %-15s\n", i, pmRef.id.c_str(), pmRef.channel.c_str(), pmRef.arch.c_str(), pmRef.module.c_str(), pmRef.version.c_str());
        i++;
    }
}

bool KMPackageManager::removeDeployFiles()
{
    std::set<std::string> allUniqueLines;

    for (auto &it : d->m_pmUninstallVector)
    {
        fs::path versionFile = it.baseInstallDir + "/" + it.channel + "/" + it.arch + "/" + it.kind + "/" + it.id + "/" + it.module + "/" + it.version;
        fs::path deployFile = versionFile / "deploy";

        if (!fs::exists(deployFile))
        {
            // std::cerr << "文件不存在: " << deployFile << std::endl; // 输出调试信息
            continue; // 跳过该文件
        }

        auto lines = KMFolder::readDeployFiles(deployFile);
        allUniqueLines.insert(lines.begin(), lines.end());
    }

    deleteFilesIfExist(allUniqueLines);
    removeEmptyDirs(allUniqueLines);

    return true;
}

void KMPackageManager::deleteHostAppFiles()
{
    for (auto &it : d->m_pmUninstallVector)
    {
        fs::path versionFile = it.baseInstallDir + "/" + it.channel + "/" + it.arch + "/" + it.kind + "/" + it.id + "/" + it.module + "/" + it.version;
        d->m_hostAppDeploy->setHostAppConfig(versionFile, it.id);
        if (d->m_hostAppDeploy->loadHostAppInfo())
        {
            d->m_hostAppDeploy->uninstall();
        }

        clearCacheFiles(it.id);
    }
}

void KMPackageManager::removeEmptyDirs(const std::set<std::string> &files)
{
    for (const auto &file : files)
    {
        fs::path dir = fs::path(file).parent_path();
        while (dir != "/" && fs::exists(dir) && fs::is_empty(dir))
        {
            fs::remove(dir);
            KMDebug("Removed empty dir: " + dir.string());
            dir = dir.parent_path();
        }
    }
}

std::unordered_map<std::string, KMInfoJson> KMPackageManager::loadAllInfoJsonFiles(std::string pathStr)
{
    // fs::path layersPath = d->m_folder->getLayersPath();
    std::unordered_map<std::string, KMInfoJson> infoMap;

    // 遍历目录中的所有info.json文件
    try
    {
        for (const auto &dirEntry : std::filesystem::recursive_directory_iterator(pathStr))
        {
            if (dirEntry.is_regular_file() && dirEntry.path().filename() == "info.json")
            {
                auto filenameStr = dirEntry.path().string();
                KMInfoJson infoJson;
                infoJson.loadFile(filenameStr);

                size_t lastSlashPos = filenameStr.find_last_of('/');
                if (lastSlashPos != std::string::npos)
                {
                    filenameStr = filenameStr.substr(0, lastSlashPos);
                }
                // 使用文件路径作为key存储每个infoJson对象
                infoMap[filenameStr] = infoJson;
            }
        }
    }
    catch (const std::filesystem::filesystem_error &e)
    {
        std::cerr << e.what() << std::endl;
        kmlogger.perror(e.what());
    }

    return infoMap;
}

bool KMPackageManager::printAssumeUninstallYesChoice(PackageManagerOptions pmOptions)
{
    if (!pmOptions.assumeYes)
    {
        while (true)
        {
            // yesOption = std::cin.get();
            std::cout << _("Enter number to delete (separated by spaces) or 'y' to delete all, enter 'n' to cancel: ");
            d->m_pmUninstallVector.clear();
            std::string input;
            std::getline(std::cin, input);

            // 检查是否输入 'a'
            if (input == "y" || input == "Y")
            {
                d->m_pmUninstallVector = d->m_pmQueryVector;
                // pmOptions.assumeYes = true;
                break;
            }

            if (input == "n" || input == "N")
            {
                return false;
            }

            std::istringstream iss(input);
            std::vector<int> indices;
            int index;
            bool isValid = true;

            if (input.empty())
            {
                std::cout << "Input is empty" << std::endl;
                continue;
            }

            for (char ch : input)
            {
                if (!std::isdigit(ch) && ch != ' ')
                {
                    std::cout << "Invalid idx: " << input << std::endl;
                    isValid = false;
                    break;
                }
            }

            if (!isValid)
            {
                continue;
            }

            while (iss >> index)
            {
                indices.push_back(index);
            }

            indices.erase(std::unique(indices.begin(), indices.end()), indices.end());

            for (int idx : indices)
            {
                if (idx > 0 && idx <= static_cast<int>(d->m_pmQueryVector.size()))
                {
                    d->m_pmUninstallVector.push_back(d->m_pmQueryVector[idx - 1]);
                }
                else
                {
                    std::cout << "Invalid index: " << idx << std::endl;
                    isValid = false;
                }
            }

            if (!isValid)
            {
                continue;
            }

            break;
        }
    }
    else
    {
        d->m_pmUninstallVector = d->m_pmQueryVector;
    }

    return true;
}

vector<KMRef> KMPackageManager::reserveVersion()
{
    vector<KMRef> unRemove = d->m_pmQueryVector;
    for (auto &it : d->m_pmUninstallVector)
    {
        unRemove.erase(std::remove_if(unRemove.begin(), unRemove.end(), [&](const KMRef &nowIt)
                                      { return it.arch == nowIt.arch && it.module == nowIt.module && it.version == nowIt.version; }),
                       unRemove.end());
    }
    return unRemove;
}

bool KMPackageManager::removeDeployFiles(const std::string &name)
{
    std::vector<fs::path> idPaths;

    fs::path rootDir = d->m_folder->getLayersPath();
    fs::path develDir = fs::path(KMStorageDir::getUserBaseDirLocation());

    KMFolder::findIdPaths(rootDir, name, idPaths);
    if (fs::exists(develDir))
    {
        KMFolder::findIdPaths(develDir, name, idPaths);
    }
    if (idPaths.empty())
    {
        // std::cerr << "No ID paths found for ID: " << name << std::endl;
        return false;
    }

    std::cout << "ID paths found: " << std::endl;
    for (const auto &path : idPaths)
    {
        std::cout << "  " << path << std::endl;
    }

    // Step 2: Find deploy files in each ID path
    std::vector<fs::path> deployFiles;
    for (const auto &idPath : idPaths)
    {
        KMFolder::findDeployFiles(idPath, deployFiles);
    }

    if (deployFiles.empty())
    {
        std::cerr << "No deploy files found for ID: " << name << std::endl;
        return false;
    }

    std::cout << "Deploy files found: " << std::endl;
    for (const auto &file : deployFiles)
    {
        std::cout << "  " << file << std::endl;
    }

    // Step 3: Read deploy files and collect unique lines
    std::set<std::string> allUniqueLines;
    for (const auto &deployFile : deployFiles)
    {
        auto lines = KMFolder::readDeployFiles(deployFile);
        allUniqueLines.insert(lines.begin(), lines.end());
    }

    // Step 4: Output and delete unique lines
    deleteFilesIfExist(allUniqueLines);

    return true;
}

bool KMPackageManager::removeTmpDeployFiles(const std::string &name)
{
    std::vector<fs::path> idPaths;

    fs::path rootDir = fs::path("/tmp");
    fs::path develDir = fs::path(KMStorageDir::getUserBaseDirLocation());

    KMFolder::findTmpIdPaths(rootDir, name, idPaths);

    if (idPaths.empty())
    {
        kmlogger.info("No ID paths found for ID: %s", name.c_str());
        return false;
    }

    kmlogger.info("ID paths found:");
    for (const auto &path : idPaths)
    {
        kmlogger.info("  %s", path.c_str());
    }

    for (const auto &path : idPaths)
    {
        if (fs::exists(path))
        {
            kmlogger.info("Deleting: %s", path.c_str());
            fs::remove_all(path);
        }
    }

    return true;
}

bool KMPackageManager::removeSpecificDeployFiles(const std::string &str)
{
    std::set<std::string> allUniqueLines;
    fs::path versionFile = str + "/" + "deploy";

    if (!fs::exists(versionFile))
    {
        // std::cerr << "文件不存在: " << deployFile << std::endl; // 输出调试信息
        return false; // 跳过该文件
    }

    auto lines = KMFolder::readDeployFiles(versionFile);
    allUniqueLines.insert(lines.begin(), lines.end());

    deleteFilesIfExist(allUniqueLines);
    removeEmptyDirs(allUniqueLines);

    return true;
}

void KMPackageManager::deleteFilesIfExist(const std::set<std::string> &fileList)
{
    for (const auto &filePath : fileList)
    {
        try
        {
            if (fs::exists(filePath))
            {
                std::cout << "File exists: " << filePath << ". Deleting..." << std::endl;
                fs::remove_all(filePath);
            }
            else
            {
                // std::cout << "File does not exist: " << filePath << ". Skipping..." << std::endl;
            }
        }
        catch (const fs::filesystem_error &e)
        {
            std::cerr << "Error processing file: " << filePath << ". Error: " << e.what() << std::endl;
        }
    }
}

bool KMPackageManager::deleteRefsFromRepo(const std::string &name)
{
    // 从本地repo中删除引用
    for (auto pair : d->m_mapRefs)
    {
        if (pair.second.id == name)
        {
            string refString = pair.second.branch + "/" + pair.second.arch + "/" + pair.second.kind + "/" + pair.second.id + "/" + pair.second.module + "/" + pair.second.version;

            if (!d->m_ostreeHandler->deleteRefTag("", string("deploy/").append(refString)))
            {
                return false;
            }

            if (!d->m_ostreeHandler->deleteRefTag(BASE_REOMTE, refString)) // file:///opt/kaiming/repo/refs/remotes/kaiming-repo/stable/x86_64/app/top.openkylin.Music
            {
                return false;
            }

            if (!d->m_ostreeHandler->repoPrune())
            {
                return false;
            }
        }
    }
    return true;
}

bool KMPackageManager::deleteSpecificRefsFromRepo(const std::string &str)
{
    int count = 0;
    size_t pos = str.length(); // 从末尾开始
    std::string extractedStr;
    // 从后往前找第 6 个 '/'
    while (pos != 0)
    {
        pos--;
        if (str[pos] == '/')
        {
            count++;
            if (count == 6)
            {
                extractedStr = str.substr(pos + 1); // 返回第 6 个 '/' 之后的内容
            }
        }
    }

    // 从本地repo中删除引用
    for (auto pair : d->m_mapRefs)
    {
        string refString = pair.second.branch + "/" + pair.second.arch + "/" + pair.second.kind + "/" + pair.second.id + "/" + pair.second.module + "/" + pair.second.version;

        if (refString == extractedStr)
        {
            if (!d->m_ostreeHandler->deleteRefTag("", string("deploy/").append(refString)))
            {
                return false;
            }

            if (!d->m_ostreeHandler->deleteRefTag(BASE_REOMTE, refString)) // file:///opt/kaiming/repo/refs/remotes/kaiming-repo/stable/x86_64/app/top.openkylin.Music
            {
                return false;
            }

            if (!d->m_ostreeHandler->repoPrune())
            {
                return false;
            }
            break;
        }
    }

    return true;
}

bool KMPackageManager::deleteUserData(const std::string &name)
{
    printf("remove app %s user data dir.\n", name.c_str());

    if (!fs::remove_all(d->m_folder->getAppDataPath(name)))
    {
        kmlogger.error("uninstall %s error, can't delete the directory.", name);
        return false;
    }
    return true;
}

int KMPackageManager::update(const vector<string> &vectorIds, PackageManagerOptions pmOptions)
{
    // 下载和解析清单依赖文件 summary.json
    std::cout << _("Search matching packages ...") << std::endl;

    fs::path basePath = d->m_folder->getPath();
    fs::path summaryJsonPath = basePath / "summary.json";
    string ostreeRepoUrl = d->m_ostreeHandler->getRemoteUrl(REPO_NAME);

    if (KMFileUtils::isFileExpired(summaryJsonPath.string(), SUMMARY_EXPIRE_SECONDS))
    {
        fs::path summaryUrl = fs::path(ostreeRepoUrl) / "summary.json";
        int getSummaryState = getRemoteSummary(summaryUrl.string(), true);

        if (getSummaryState != KM_ERROR_NO)
        {
            kmlogger.error("Get and parse summary.json file error!");
            return getSummaryState;
        }
    }

    if (vectorIds.empty())
    {
        // 仅更新包依赖清单文件
        return KM_ERROR_NO;
    }

    // 遍历取得所有符合条件的依赖列表
    int getFuzzyMatchState = getAllFuzzyMatchRefVector(vectorIds, pmOptions, TASK_TYPE_UPDATE);
    if (getFuzzyMatchState != KM_ERROR_NO)
    {
        kmlogger.error("Get fuzzy match ref error!");
        return getFuzzyMatchState;
    }
    getUsefulRefList();
    // 获取真正更新列表
    getDownloadRefList(TASK_TYPE_UPDATE);

    if (d->m_pmRefDownloadList.empty())
    {
        printf("%s\n", _("Not need to update!"));
        return KM_ERROR_NOT_NEED_UPDATE;
    }

    // 展示下载列表信息
    printDownloadRefList(d->m_pmRefDownloadList);

    // 用户选择是否展示信息
    if (!printAssumeYesChoice(pmOptions))
    {
        return KM_ERROR_NOT_INSTALLED;
    };

    // 检测当前网址是否可以访问
    if (KMStringUtils::contains(ostreeRepoUrl, "http"))
    {
        if (KMCurl::checkURL(ostreeRepoUrl, 3))
        {
            kmlogger.debug("URL is accessible!");
        }
        else
        {
            kmlogger.error("Failed to access URL!");
            return KM_ERROR_CURL_CHECK_URL_FAILED;
        }
    }

    // 下载包
    int pullFlag = false;
    pullFlag = d->m_ostreeHandler->pull(REPO_NAME, d->m_pmDownloadRefs, progressCallback);
    if (!pullFlag)
    {
        return KM_ERROR_PULL_FAILED;
    }
    for (auto &ref : d->m_pmOldAVersion)
    {
        if (!removeSpecificDeployFiles(ref))
        {
            return KM_ERROR_DELETE_DIRECTORY_FAILED;
        }
    }
    // 部署包
    return deploy();
}

int KMPackageManager::filterIdsNotInRemote(PackageManagerOptions pmOptions)
{
    std::map<std::string, std::string> localAppsInfo = d->m_list->getLocalAppIds(false);

    // 下载和解析清单依赖文件 summary.json
    fs::path basePath = d->m_folder->getPath();
    fs::path summaryJsonPath = basePath / "summary.json";

    if (KMFileUtils::isFileExpired(summaryJsonPath.string(), SUMMARY_EXPIRE_SECONDS))
    {
        string ostreeRepoUrl = d->m_ostreeHandler->getRemoteUrl(REPO_NAME);
        fs::path summaryUrl = fs::path(ostreeRepoUrl) / "summary.json";

        int getSummaryState = getRemoteSummary(summaryUrl.string(), true);
        if (getSummaryState != KM_ERROR_NO)
        {
            kmlogger.error("Get and parse summary.json file error!");
            return getSummaryState;
        }
    }

    pmOptions.arch = KMUtil::kmGetArch();

    for (const auto &app : localAppsInfo)
    {
        string completePath = d->m_parseSummaryJson->printSearchJson(app.first, pmOptions.arch, pmOptions.kind, pmOptions.module, pmOptions.isPrint, pmOptions.isGetDevel);

        if (!completePath.empty())
        {
            kmlogger.info("Currently installed app %s , completePath is : %s", app.first.c_str(), completePath.c_str());

            std::vector<std::string> splitPath;
            std::istringstream iss(completePath);
            std::string token;
            while (std::getline(iss, token, '/'))
            {
                if (!token.empty())
                {
                    splitPath.push_back(token);
                }
            }

            std::string remoteArch, remoteId, remoteVersion;
            if (splitPath.size() >= 8)
            {
                remoteArch = splitPath[1];
                remoteId = splitPath[3];
                remoteVersion = splitPath[5];
            }

            KMVersion parseVersion = KMVersion::parse(remoteVersion, true);
            KMVersion appVersion = KMVersion::parse(app.second, true);
            if (!remoteId.empty() && appVersion < parseVersion)
            {
                d->m_upgragableList.push_back({remoteId, remoteVersion, remoteArch});
            }
        }
    }

    return KM_ERROR_NO;
}

int KMPackageManager::getUpgradableList(const std::map<std::string, std::string> &localAppsInfo, PackageManagerOptions pmOptions)
{
    if (!pmOptions.upgradable)
    {
        // 环境校验
        if (!d->m_env->passEnvCheck())
        {
            kmlogger.error("Failed to check environments!");
            return KM_ERROR_CHECK_ENV_FAILED;
        }

        d->m_folder->ensureRepoForSudo(false);
    }

    std::vector<std::string> vectorIds;
    for (const auto &ids : localAppsInfo)
    {
        vectorIds.push_back(ids.first);
    }

    fs::path basePath = d->m_folder->getPath();

    std::map<std::string, std::string> localInfo = localAppsInfo;
    int errorCode = filterIdsNotInRemote(pmOptions);
    if (errorCode != KM_ERROR_NO)
    {
        return errorCode;
    }
    else
    {
        if (!pmOptions.upgradable)
        {
            std::cout << _("There are ") << d->m_upgragableList.size() << _(" packages available for upgrade. Please run 'kaiming update --upgradable' to check them.") << std::endl;

            if (!d->m_upgragableList.empty())
            {
                fs::path cachePath = basePath / "cache";
                if (!fs::exists(cachePath))
                {
                    fs::create_directories(cachePath);
                }

                fs::path upgradablePath = cachePath / "upgradable.list";
                std::ofstream outfile(upgradablePath);

                if (outfile.is_open())
                {
                    for (const auto &vec : d->m_upgragableList)
                    {
                        kmlogger.info("%s is upgradable.", vec.id.c_str());
                        outfile << vec.id << ":" << vec.version << std::endl;
                    }
                    outfile.close();
                }
                else
                {
                    kmlogger.info("Failed to create %s", upgradablePath.string());
                    return KM_ERROR_CREATE_FILE_FAILED;
                }
            }
        }
        else
        {
            for (const auto &vec : d->m_upgragableList)
            {
                std::string version = localInfo[vec.id];
                std::cout << vec.id << " " << vec.version << " " << vec.arch << " [" << _("Can be upgraded from this version: ") << version << "]" << std::endl;
            }
        }
    }
    d->m_upgragableList.clear();
    return KM_ERROR_NO;
}

int KMPackageManager::dbusSearch(const string &strID, PackageManagerOptions pmOptions)
{
    GError *error = NULL;
    // 连接到 D-Bus 系统总线
    GDBusConnection *connection = g_bus_get_sync(G_BUS_TYPE_SYSTEM, NULL, &error);
    if (!connection)
    { // 连接失败
        kmlogger.error("Failed to connect to D-Bus: %s", error->message);
        g_clear_error(&error);
    }

    // 调用 getRemoteSummary 方法
    GVariant *result = g_dbus_connection_call_sync(
        connection,
        "org.kylin.kaiming",           // 服务名称
        "/org/kylin/kaiming",          // 对象路径
        "org.kylin.kaiming.interface", // 接口名称
        "getRemoteSummary",            // 方法名  与xml中的方法名一致
        NULL,                          // 传递给 getRemoteSummary 方法的参数
        // g_variant_new("(s)", vectorIds.c_str()), // 参数元组
        NULL, // 返回值类型（NULL 表示不检查）
        G_DBUS_CALL_FLAGS_NONE,
        -1,   // 超时时间,-1 表示默认超时
        NULL, // Cancellable
        &error);

    if (error)
    { // 调用失败
        kmlogger.error("Failed to call getRemoteSummary: %s", error->message);
        std::cout << _("Failed to call getRemoteSummary") << std::endl;
        return KM_ERROR_CURL_DOWNLOAD_FAILED;
        g_clear_error(&error);
    }
    else
    {
        gboolean success;
        g_variant_get(result, "(b)", &success); // 假设返回值是布尔值
        kmlogger.debug("getRemoteSummary method call result: %s", success ? "Success" : "Failure");
        if (success == false)
        {
            std::cout << _("Please check your network connection!") << std::endl;
            return KM_ERROR_CURL_DOWNLOAD_FAILED;
        }
        g_variant_unref(result); // 释放资源
    }

    g_object_unref(connection); // 释放资源

    fs::path basePath = d->m_folder->getPath();
    fs::path summaryJsonPath = basePath / "summary.json";
    d->m_parseSummaryJson->loadFile(summaryJsonPath);
    string completePath = d->m_parseSummaryJson->printSearchJson(strID, pmOptions.arch, pmOptions.kind, pmOptions.module, pmOptions.isPrint, pmOptions.isGetDevel);

    return KM_ERROR_NO;
}

int KMPackageManager::search(const string &strID, PackageManagerOptions pmOptions)
{
    std::cout << _("Search matching packages ...") << std::endl;
    fs::path basePath = d->m_folder->getPath();
    fs::path summaryJsonPath = basePath / "summary.json";

    if (KMFileUtils::isFileExpired(summaryJsonPath.string(), SUMMARY_EXPIRE_SECONDS))
    {
        string ostreeRepoUrl = d->m_ostreeHandler->getRemoteUrl(REPO_NAME);
        fs::path summaryUrl = fs::path(ostreeRepoUrl) / "summary.json";

        int getSummaryState = getRemoteSummary(summaryUrl.string(), true);
        if (getSummaryState != KM_ERROR_NO)
        {
            kmlogger.error("Get and parse summary.json file error!");
            return getSummaryState;
        }
    }

    d->m_parseSummaryJson->loadFile(summaryJsonPath.string());
    string completePath = d->m_parseSummaryJson->printSearchJson(strID, pmOptions.arch, pmOptions.kind, pmOptions.module, pmOptions.isPrint, pmOptions.isGetDevel);

    return KM_ERROR_NO;
}

int KMPackageManager::searchInfo(const string &strID, PackageManagerOptions pmOptions, std::string &strCompletePath)
{
    if (pmOptions.isPrint)
    {
        std::cout << _("Search matching packages ...") << std::endl;
    }

    fs::path basePath = d->m_folder->getPath();
    fs::path summaryJsonPath = basePath / "summary.json";

    if (KMFileUtils::isFileExpired(summaryJsonPath.string(), SUMMARY_EXPIRE_SECONDS))
    {
        string ostreeRepoUrl = d->m_ostreeHandler->getRemoteUrl(REPO_NAME);
        fs::path summaryUrl = fs::path(ostreeRepoUrl) / "summary.json";

        int getSummaryState = getRemoteSummary(summaryUrl.string(), true);
        if (getSummaryState != KM_ERROR_NO)
        {
            kmlogger.error("Get and parse summary.json file error!");
            return getSummaryState;
        }
    } else {
        d->m_parseSummaryJson->loadFile(summaryJsonPath.string()); // for system-dus restart load
    }

    pmOptions.arch = KMUtil::kmGetArch();
    strCompletePath = d->m_parseSummaryJson->printSearchJson(strID, pmOptions.arch, pmOptions.kind, pmOptions.module, pmOptions.isPrint, pmOptions.isGetDevel);

    return KM_ERROR_NO;
}

int KMPackageManager::repair(const vector<string> &vectorIds, PackageManagerOptions pmOptions)
{
    vector<KMPMRef>  installVector;

    for (auto name : vectorIds)
    {
        kmlogger.debug("Install app %s", name.c_str());
        bool isMatchedRefFlag = false;

        KMPMRef pmRef = d->m_util->parseKMPMRef(name);
        d->m_util->printKMPMRef(name, pmRef);

        // 如果用户指定了 arch，则覆盖默认的 arch
        if (!pmOptions.arch.empty())
        {
            pmRef.ref.arch = pmOptions.arch;
        }

        if (pmRef.refSourceType == KMPACKAGE_SOURCE_TYPE_OK)
        {
            return KM_ERROR_REPAIR_FAILED;
        }

        if (!currentRefIsUninstalled(pmRef))
        {
            std::cout << _("Not install") << name << std::endl;
        }
    }
    
    if (d->m_pmQueryVector.empty())
    {
        return KM_ERROR_REPAIR_FAILED;
    }
    // 完整的 ref ，如 "stable/x86_64/app/top.openkylin.Music/binary/2.0.0.0"


    // 进行损坏检测，过滤掉未损坏的应用
    auto it = d->m_pmQueryVector.begin();
    while (it != d->m_pmQueryVector.end())
    {
        KMRef ref = *(it);

        // 对修复的应用包files做md5判断，如果计算得到的md5与原md5一致则该包不损坏，跳过修复
        std::string layersPath = ref.baseInstallDir;

        std::string completedRef = ref.channel + '/' + ref.arch + '/' + ref.kind + '/' + ref.id + '/' + ref.module + '/' + ref.version;
        // 计算出files的md5
        std::string filesDirPath = KMStringUtils::buildFilename(layersPath, completedRef);
        std::string filesMd5 = KMStringUtils::buildFilename(filesDirPath, "files.md5");
        std::string filePrefix; // 读取出本地md5文件的值

        // 读取md5的文件不存在，该包存在修复列表中
        if (!fs::exists(filesMd5))
        {
            KMPMRef installRef;
            installRef.ref = ref;
            installRef.ref.version = ref.version[0];
            installVector.push_back(installRef);
            ++it; //迭代计算下一个文件
        }
        else
        {
            if (KMFileUtils::isObsolete(filesMd5))
            {
                std::string filesDirFiles = KMStringUtils::buildFilename(filesDirPath, "files");
                std::cout << _("Verify file MD5 checksum ") << ref.id << endl;
                std::string checksum = KMFileUtils::getDirMd5sum(filesDirFiles);

                //读取之前判断md5是否存在
                KMFileUtils::readFile(filesMd5, filePrefix);

                // 如果本地的Md5的文件与计算出来的Md5文件相同，则该app/base/runtime则不需要修复
                if ((filePrefix == checksum) && (checksum != ""))
                {
                    it = d->m_pmQueryVector.erase(it); // 修复列表中移除当前app/base/runtime
                    std::cout << _("Repair") << ref.id << _("complete") << endl;
                }
                else
                {
                    KMPMRef installRef;
                    installRef.ref = ref;
                    installRef.ref.version = ref.version[0];
                    installVector.push_back(installRef);
                    ++it; //迭代计算下一个文件
                }
            }
            else
            {
                KMDirMd5sum md5sum(filesDirPath);
                if (md5sum.checksum("files"))
                {
                    it = d->m_pmQueryVector.erase(it); // 修复列表中移除当前app/base/runtime
                    std::cout << _("Repair") << ref.id << _("complete") << endl;
                }
                else
                {
                    KMPMRef installRef;
                    installRef.ref = ref;
                    installRef.ref.version = ref.version[0];
                    installVector.push_back(installRef);
                    ++it; //迭代计算下一个文件
                }
            }
        }
    }

    //如果得到的修复list为空说明不需要修复则直接返回
    if (d->m_pmQueryVector.empty())
    {
        return KM_ERROR_NO;
    }
    
    fs::path repoPath = d->m_folder->getRepoPath(); // /opt/kaiming/repo
    string ostreeRepoUrl = d->m_ostreeHandler->getRemoteUrl(REPO_NAME);
    std::string summaryUrl = ostreeRepoUrl.append("/").append("summary.json");
    fs::path basePath = d->m_folder->getPath();
    fs::path summaryJsonPath = basePath / "summary.json";
    bool isSummaryJsonExpired = KMFileUtils::isFileExpired(summaryJsonPath, 3600);
    int getSummaryState = getRemoteSummary(summaryUrl, isSummaryJsonExpired);

    if (getSummaryState != KM_ERROR_NO)
    {
        kmlogger.error("Get and parse summary.json file error!");
        return getSummaryState;
    }

    //下载损坏文件
    for (auto findIt = installVector.begin(); findIt!= installVector.end();)
    {
        KMPMRef pmRef = *findIt;
        vector<string> fuzzyMatchRefs;
        // 从远程依赖文件查找对应的包是否存在，并获取所有的依赖列表
        d->m_parseSummaryJson->fuzzyMatchCompletedRef(pmRef, fuzzyMatchRefs);
        string notInstallable = d->m_parseSummaryJson->getNotInstallable();
        vector<std::string> notFind = d->m_parseSummaryJson->getNotFind();

        if (!notInstallable.empty())
        {
            if (pmRef.ref.id == notInstallable)
            {
                std::cout << _("Can not install this package ") << pmRef.ref.id << std::endl;
            }
            else
            {
                std::cout << pmRef.ref.id << _(" Can not install the dependencies of this package:") << notInstallable  << std::endl;
            }
            findIt = installVector.erase(findIt);
            d->m_parseSummaryJson->clearNotInstallable();
            continue;
        }

        if (!notFind.empty())
        {
            if (notFind[0] == pmRef.ref.id )
            {
                std::cout<< _("Not find this online package ") << pmRef.ref.id << std::endl;
            }
            else
            {
                std::cout << pmRef.ref.id << _(" Can not find the dependencies of this package:") << std::endl;
                for (auto it : notFind)
                {
                    if (it == pmRef.ref.id)
                    {
                        continue;
                    }

                    std::cout << it << std::endl;
                }
            }
            findIt = installVector.erase(findIt);  
            d->m_parseSummaryJson->clearNotInstallable();
            continue;
        }

        findIt ++;
        d->m_pmFuzzyMatchRefs.insert(d->m_pmFuzzyMatchRefs.end(), fuzzyMatchRefs.begin(), fuzzyMatchRefs.end());
    }
    
    KMUtil::removeDuplicateValues(d->m_pmFuzzyMatchRefs);
    // 获取必要安装列表
    getUsefulRefList();
    
    for (auto &ref : d->m_pmFuzzyMatchRefs)
    {
        KMPMRef pmFuzzyRef = KMUtil::parseSummaryFuzzyKMPMRef(ref);
        pmFuzzyRef.completedRef = KMUtil::dupKMPMCompletedRef(pmFuzzyRef);
        // d->m_pmDownloadRefs.clear();
        d->m_pmDownloadRefs.push_back(pmFuzzyRef.completedRef);
        d->m_parseSummaryJson->getSummaryInfo(pmFuzzyRef);
        d->m_pmRefDownloadList.push_back(pmFuzzyRef);
        globalRefDownloadListForProgress.push_back(pmFuzzyRef);
    }

    if (KMStringUtils::contains(ostreeRepoUrl, "http"))
    {
        if (KMCurl::checkURL(ostreeRepoUrl, 3))
        {
            kmlogger.debug("URL is accessible!");
        }
        else
        {
            kmlogger.error("Failed to access URL!");
            return KM_ERROR_CURL_CHECK_URL_FAILED;
        }
    }

    // 下载包
    int pullFlag = false;
    pullFlag = d->m_ostreeHandler->pull(REPO_NAME, d->m_pmFuzzyMatchRefs, progressCallback);
    if (!pullFlag)
    {
        return KM_ERROR_PULL_FAILED;
    }

    //删除损坏文件
    for (auto &deleteRef : installVector)
    {
        KMRef ref= deleteRef.ref;
        string removeStr = ref.baseInstallDir + "/" + ref.channel + "/" + ref.arch + "/" + ref.kind + "/" + ref.id + "/" + ref.module + "/" + ref.version;
        removeSpecificDeployFiles(removeStr);
        deleteSpecificRefsFromRepo(removeStr);
    }

    //删除旧版本文件
    for (auto &ref : d->m_pmOldAVersion)
    {
        if (!removeSpecificDeployFiles(ref))
        {
            return KM_ERROR_DELETE_DIRECTORY_FAILED;
        }
        if (!d->m_pmIsForceInstall)
        {
            if (!deleteSpecificRefsFromRepo(ref))
            {
                return KM_ERROR_DELETE_REF_TAG_FAILED;
            }
        }
    }

    // 部署包
    deploy();
    for (const auto &name : vectorIds)
    {
        printf("Repair  %s complete", name.c_str());
    }
    return KM_ERROR_NO;
}

bool KMPackageManager::currentRefIsInstalled(KMPMRef &pmRef)
{
    KMInstalledAppQuery query;
    bool isInstalled = false;
    std::vector<KMRef> queryVector;
    string kindStr = pmRef.ref.kind.empty() ? "" : pmRef.ref.kind;
    string versionStr = pmRef.ref.version.empty() ? "" : pmRef.ref.version;
    string channelStr = pmRef.ref.channel.empty() ? "" : pmRef.ref.channel;
    string archStr = pmRef.ref.arch.empty() ? "" : pmRef.ref.arch;
    string moduleStr = pmRef.ref.module.empty() ? "binary" : pmRef.ref.module;

    queryVector = query.kind(kindStr).id(pmRef.ref.id).arch(archStr).channel(channelStr).version(versionStr).module(moduleStr).query(KMInstalledAppQuery::All);

    KMVersion installedVersion;
    KMVersion currentVersion;

    if (!queryVector.empty())
    {
        if (versionStr.empty())
        {
            // 未指定版本，则认为已安装
            isInstalled = true;
        }
        else
        {
            currentVersion = KMVersion::parse(versionStr, false);

            // 指定版本，则检查是否已安装
            for (const auto &ref : queryVector)
            {
                installedVersion = KMVersion::parse(ref.version, false);
                if (installedVersion == currentVersion)
                {
                    isInstalled = true;
                    break;
                }
            }
        }
    }
    return isInstalled;
}

bool KMPackageManager::currentRefVersionIsInstalled(KMPMRef &pmRef)
{
    KMInstalledAppQuery query;
    bool isInstalled = false;
    std::vector<KMRef> queryVector;
    string kindStr = pmRef.ref.kind.empty() ? "" : pmRef.ref.kind;
    string versionStr = pmRef.ref.version.empty() ? "" : pmRef.ref.version;
    string channelStr = pmRef.ref.channel.empty() ? "" : pmRef.ref.channel;
    string archStr = pmRef.ref.arch.empty() ? "" : pmRef.ref.arch;
    string moduleStr = pmRef.ref.module.empty() ? "binary" : pmRef.ref.module;

    queryVector = query.kind(kindStr).id(pmRef.ref.id).arch(archStr).channel(channelStr).version(versionStr).module(moduleStr).query(KMInstalledAppQuery::All);

    KMVersion installedVersion;
    KMVersion currentVersion;

    if (!queryVector.empty())
    {
        if (versionStr.empty())
        {
            // 未指定版本，则认为已安装
            // isInstalled = true;
        }
        else
        {
            currentVersion = KMVersion::parse(versionStr, false);

            // 指定版本，则检查是否已安装
            for (const auto &ref : queryVector)
            {
                installedVersion = KMVersion::parse(ref.version, false);
                if (installedVersion == currentVersion)
                {
                    isInstalled = true;
                    break;
                }
            }
        }
    }
    return isInstalled;
}

bool KMPackageManager::currentRefIsUninstalled(KMPMRef &pmRef)
{
    KMInstalledAppQuery query;
    bool isInstalled = false;
    // std::vector<KMRef> queryVector;
    string kindStr = pmRef.ref.kind.empty() ? "" : pmRef.ref.kind;
    string versionStr = "";
    // string versionStr = pmRef.ref.version.empty() ? "" : pmRef.ref.version;
    string channelStr = pmRef.ref.channel.empty() ? "" : pmRef.ref.channel;
    string moduleStr = pmRef.ref.module.empty() ? "" : pmRef.ref.module;
    string archStr = pmRef.ref.arch.empty() ? "" : pmRef.ref.arch;

    d->m_pmQueryVector = query.kind(kindStr).id(pmRef.ref.id).arch(archStr).channel(channelStr).version(versionStr).module(moduleStr).query(KMInstalledAppQuery::All);

    KMVersion installedVersion;
    KMVersion currentVersion;

    if (!d->m_pmQueryVector.empty())
    {
        if (versionStr.empty())
        {
            // 未指定版本，则认为已安装
            isInstalled = true;
        }
        else
        {
            currentVersion = KMVersion::parse(versionStr, false);

            // 指定版本，则检查是否已安装
            for (const auto &ref : d->m_pmQueryVector)
            {
                installedVersion = KMVersion::parse(ref.version, false);
                if (installedVersion == currentVersion)
                {
                    isInstalled = true;
                    break;
                }
            }
        }
    }
    return isInstalled;
}

bool KMPackageManager::currentFuzzyRefIsInstalled(KMPMRef &pmRef)
{
    KMInstalledAppQuery query;
    bool isInstalled = false;
    string kindStr = pmRef.ref.kind.empty() ? "" : pmRef.ref.kind;
    string versionStr = "";
    string channelStr = pmRef.ref.channel.empty() ? "" : pmRef.ref.channel;
    string moduleStr = pmRef.ref.module.empty() ? "binary" : pmRef.ref.module;
    string archStr = pmRef.ref.arch.empty() ? "" : pmRef.ref.arch;

    std::vector<KMRef> queryVector = query.kind(kindStr).id(pmRef.ref.id).arch(archStr).channel(channelStr).version(versionStr).module(moduleStr).query(KMInstalledAppQuery::All);

    KMVersion installedVersion;
    KMVersion currentVersion = KMVersion::parse(pmRef.ref.version, false);

    if (!queryVector.empty())
    {
        // 指定版本，则检查是否已安装
        for (const auto &ref : queryVector)
        {
            installedVersion = KMVersion::parse(ref.version, false);
            if (ref.kind == "base" || ref.kind == "runtime" || ref.kind == "depend")
            {
                if (installedVersion.getMajor() == currentVersion.getMajor())
                {
                    if (installedVersion.getMinor() == currentVersion.getMinor())
                    {
                        if (installedVersion.getPatch() == currentVersion.getPatch())
                        {
                            if (d->m_pmIsForceInstall)
                            {
                                string removeStr = ref.baseInstallDir + "/" + ref.channel + "/" + ref.arch + "/" + ref.kind + "/" + ref.id + "/" + ref.module + "/" + ref.version;
                                d->m_pmOldAVersion.push_back(removeStr);
                            }
                            else
                            {
                                int maxVersion = installedVersion.compare(currentVersion);
                                if (maxVersion == 0)
                                {
                                    isInstalled = true;
                                }
                                if (maxVersion == -1)
                                {
                                    string removeStr = ref.baseInstallDir + "/" + ref.channel + "/" + ref.arch + "/" + ref.kind + "/" + ref.id + "/" + ref.module + "/" + ref.version;
                                    d->m_pmOldAVersion.push_back(removeStr);
                                }
                                if (maxVersion == 1 && pmRef.ref.isSpecialVersion != true)
                                {
                                    isInstalled = true;
                                }
                            }
                            break;
                        }
                        else
                        {
                            if (installedVersion.getPatch() < currentVersion.getPatch())
                            {
                                string removeStr = ref.baseInstallDir + "/" + ref.channel + "/" + ref.arch + "/" + ref.kind + "/" + ref.id + "/" + ref.module + "/" + ref.version;
                                d->m_pmOldAVersion.push_back(removeStr);
                                continue;
                            }
                            if (installedVersion.getPatch() > currentVersion.getPatch() && pmRef.ref.isSpecialVersion != true)
                            {
                                isInstalled = true;
                            }
                        }
                    }
                    else if (installedVersion.getMinor() < currentVersion.getMinor())
                    {
                        string removeStr = ref.baseInstallDir + "/" + ref.channel + "/" + ref.arch + "/" + ref.kind + "/" + ref.id + "/" + ref.module + "/" + ref.version;
                        d->m_pmOldAVersion.push_back(removeStr);
                        continue;
                    }
                    else
                    {
                        if (pmRef.ref.isSpecialVersion != true)
                        {
                            isInstalled = true;
                        }
                    }
                }
            }
            else
            {
                std::string idVersion = pmRef.ref.id + pmRef.ref.version;
                auto it = std::find(d->m_pmSpecialIdVersion.begin(), d->m_pmSpecialIdVersion.end(), idVersion);
                bool isSpecialVersion = false;
                if (it != d->m_pmSpecialIdVersion.end())
                {
                    isSpecialVersion = true;
                }
                if (installedVersion < currentVersion || d->m_pmIsForceInstall || isSpecialVersion == true)
                {
                    string removeStr = ref.baseInstallDir + "/" + ref.channel + "/" + ref.arch + "/" + ref.kind + "/" + ref.id + "/" + ref.module + "/" + ref.version;
                    d->m_pmOldAVersion.push_back(removeStr);
                }
                else
                {
                    isInstalled = true;
                    break;
                }
            }
        }
    }
    return isInstalled;
}

int KMPackageManager::isInstalled(string name)
{
    kmlogger.debug("Check wait for installing app %s", name.c_str());

    KMPMRef pmRef = KMUtil::parseKMPMRef(name);

    if (currentRefIsInstalled(pmRef))
    {
        kmlogger.info("%s is installed", name.c_str());
        return KM_ERROR_NO;
    }

    return KM_ERROR_NOT_INSTALLED;
}

int KMPackageManager::isUpgradabled(string name)
{
    kmlogger.debug("Check wait for update app %s", name.c_str());

    KMPMRef pmRef = KMUtil::parseKMPMRef(name);

    // 不安装,则不更新
    if (!currentRefIsInstalled(pmRef))
    {
        kmlogger.info("%s is not installed", name.c_str());
        return KM_ERROR_NOT_INSTALLED;
    }

    kmlogger.info("开始在远程仓库中查找 ref 对应的最新 commitid ...");

    // 下载和解析清单依赖文件 summary.json
    string ostreeRepoUrl = d->m_ostreeHandler->getRemoteUrl(REPO_NAME);
    fs::path summaryUrl = fs::path(ostreeRepoUrl) / "summary.json";

    int getSummaryState = getRemoteSummary(summaryUrl.string(), false);
    if (getSummaryState != KM_ERROR_NO)
    {
        kmlogger.error("Get and parse summary.json file error!");
        return getSummaryState;
    }

    // 从远程依赖文件查找对应的包是否存在，并获取所有的依赖列表
    bool isMatchedRefFlag = false;
    isMatchedRefFlag = d->m_parseSummaryJson->fuzzyMatchCompletedRef(pmRef, d->m_pmFuzzyMatchRefs);
    if (!isMatchedRefFlag)
    {
        std::cout << _("Not find this online ref ") << name << std::endl;
        return KM_ERROR_REF_NOT_FOUND;
    }

    for (const auto &ref : d->m_pmFuzzyMatchRefs)
    {
        kmlogger.debug("%s", ref.c_str());
    }

    // 依赖列表去重
    KMUtil::removeDuplicateValues(d->m_pmFuzzyMatchRefs);

    vector<string> pmMatchRefs;
    for (const auto &ref : d->m_pmFuzzyMatchRefs)
    {
        if (KMStringUtils::contains(ref, pmRef.ref.id))
        {
            pmMatchRefs.push_back(ref);
        }
    }

    // only one : "stable/x86_64/app/top.openkylin.Music/binary/2.0.1.0"
    for (const auto &ref : pmMatchRefs)
    {
        KMPMRef pmUpdateRef = KMUtil::parseSummaryFuzzyKMPMRef(ref);

        kmlogger.debug("%s", ref.c_str());

        if (!currentRefIsInstalled(pmUpdateRef))
        {
            kmlogger.info("need update %s", ref.c_str());
            return KM_ERROR_NO;
        }
    }

    printf("%s\n", _("This app not need update!"));
    return KM_ERROR_NOT_NEED_UPDATE;
}

int KMPackageManager::getRemoteSummary(const string &summaryUrl, bool isExpired)
{
    static bool hasNetworkError = false;
    fs::path basePath = d->m_folder->getPath();
    fs::path summaryJsonPath = basePath / "summary.json";
    bool needDownload = false;
    bool needReload = false;

    kmlogger.debug("Begin to curl %s", summaryUrl.c_str());

    if (!fs::exists(summaryJsonPath))
    {
        kmlogger.info("summary.json does not exist, will download");
        fs::create_directories(basePath);
        needDownload = true;
    }
    else if (isExpired)
    {
        kmlogger.info("summary.json is expired, will update");
        needDownload = true;
    }
    else if (!d->m_util->isJsonFile(summaryJsonPath.string()))
    {
        kmlogger.error("Existing summary.json is not valid, will re-download");
        needDownload = true;
    }

    // Download if needed
    if (needDownload)
    {
        if (!KMCurl::curlApiDownload(summaryUrl, summaryJsonPath.string()))
        {
            kmlogger.error("Curl download failed. SummaryUrl: %s, Path: %s", summaryUrl.c_str(), summaryJsonPath.string().c_str());
            if (!hasNetworkError)
            {
                std::cout << _("Please check your network connection!") << std::endl;
                hasNetworkError = true;
            }
            return KM_ERROR_CURL_DOWNLOAD_FAILED;
        }

        if (!d->m_util->isJsonFile(summaryJsonPath.string()))
        {
            kmlogger.error("Downloaded summary.json is not valid");
            std::cout << _("summary.json is not valid, please check repo config URL") << std::endl;
            return KM_ERROR_SUMMARY_JSON_INVALID;
        }

        needReload = true;
    }

    // Load and parse the file
    if (d->m_parseSummaryJson->loadFile(summaryJsonPath.string(), needReload) != KM_ERROR_NO)
    {
        kmlogger.error("Failed to load summary.json");
        return KM_ERROR_LOAD_FILE_FAILED;
    }

    return KM_ERROR_NO;
}

int KMPackageManager::getAllFuzzyMatchRefVector(const vector<string> &vectorIds, PackageManagerOptions pmOptions, TaskType taskType)
{
    for (const auto &name : vectorIds)
    {
        kmlogger.debug("Install app %s", name.c_str());
        bool isMatchedRefFlag = false;

        KMPMRef pmRef = d->m_util->parseKMPMRef(name);
        d->m_util->printKMPMRef(name, pmRef);

        // 如果用户指定了 arch，则覆盖默认的 arch
        if (!pmOptions.arch.empty())
        {
            pmRef.ref.arch = pmOptions.arch;
        }

        if (!pmOptions.kind.empty())
        {
            pmRef.ref.kind = pmOptions.kind;
        }

        if (!pmRef.ref.version.empty())
        {
            pmRef.ref.isSpecialVersion = true;
            d->m_pmSpecialIdVersion.push_back(pmRef.ref.id + pmRef.ref.version);
        }

        d->m_pmInstallId.push_back(pmRef.ref.id);

        switch (taskType)
        {
        case TASK_TYPE_INSTALL:
            // 已安装当前软件包或指定版本包则不再安装
            if (currentRefVersionIsInstalled(pmRef))
            {
                if (!pmOptions.force)
                {
                    std::cout << _("This package has been installed! ") << pmRef.ref.id << std::endl;
                    return KM_ERROR_FORCE_INSTALL;
                }
                kmlogger.info("当前软件包 %s 已安装", name.c_str());
                continue;
            }

            break;
        case TASK_TYPE_UPDATE:
            // 未安装该包或该包指定版本,则不更新，直接返回
            if (!currentRefIsUninstalled(pmRef))
            {
                if (pmRef.ref.version.empty())
                {
                    kmlogger.info("未安装该软件包 %s", name.c_str());
                    std::cout << _("Not install this ref ") << name << std::endl;
                    return KM_ERROR_REF_NOT_INSTALLED;
                }
            }
            break;
        case TASK_TYPE_UNINSTALL:
            // 未安装该包或该包指定版本,则不更新，直接返回
            if (!currentRefIsUninstalled(pmRef))
            {
                kmlogger.error("%s not installed!", name.c_str());
                printf("%s not installed!", name.c_str());
                return KM_ERROR_REF_NOT_INSTALLED;
            }
            break;
        case TASK_TYPE_REPAIR:
            // 未安装该包或该包指定版本,则不修复，直接返回
            if (!currentRefIsUninstalled(pmRef))
            {
                kmlogger.error("%s not installed!", name.c_str());
                printf("%s not installed!", name.c_str());
                return KM_ERROR_REF_NOT_INSTALLED;
            }
            break;
        default:
            break;
        }

        vector<string> fuzzyMatchRefs;
        // 从远程依赖文件查找对应的包是否存在，并获取所有的依赖列表
        isMatchedRefFlag = d->m_parseSummaryJson->fuzzyMatchCompletedRef(pmRef, fuzzyMatchRefs);
        string notInstallable = d->m_parseSummaryJson->getNotInstallable();
        vector<std::string> notFind = d->m_parseSummaryJson->getNotFind();
        if (!notInstallable.empty())
        {
            d->m_pmCanNotInstallVector.push_back(name);

            if (pmRef.ref.id == notInstallable)
            {
                std::cout << _("Can not install this package ") << name << std::endl;
            }
            else
            {
                std::cout << name << _(" Can not install the dependencies of this package:") << notInstallable << std::endl;
            }

            d->m_parseSummaryJson->clearNotInstallable();
            continue;
        }
        if (!notFind.empty())
        {
            d->m_pmCanNotInstallVector.push_back(name);

            if (notFind[0] == pmRef.ref.id)
            {
                std::cout << _("Not find this online package ") << name << std::endl;
            }
            else
            {
                std::cout << name << _(" Can not find the dependencies of this package:") << std::endl;
                for (auto it : notFind)
                {
                    if (it == pmRef.ref.id)
                    {
                        continue;
                    }

                    std::cout << it << std::endl;
                }
            }

            d->m_parseSummaryJson->clearNotInstallable();
            continue;
        }

        d->m_pmFuzzyMatchRefs.insert(d->m_pmFuzzyMatchRefs.end(), fuzzyMatchRefs.begin(), fuzzyMatchRefs.end());

        for (const auto &ref : d->m_pmFuzzyMatchRefs)
        {
            kmlogger.debug("%s", ref.c_str());
        }
    }

    // 依赖列表去重
    KMUtil::removeDuplicateValues(d->m_pmFuzzyMatchRefs);
    return KM_ERROR_NO;
}

void KMPackageManager::getUsefulRefList()
{
    string preRef;
    for (auto it = d->m_pmFuzzyMatchRefs.begin(); it != d->m_pmFuzzyMatchRefs.end(); it++)
    {
        string ref = *it;
        if (preRef.empty())
        {
            preRef = ref;
            continue;
        }
        else
        {
            KMPMRef pmFuzzyRef = KMUtil::parseSummaryFuzzyKMPMRef(ref);
            KMPMRef pmPreRef = KMUtil::parseSummaryFuzzyKMPMRef(preRef);
            if (pmFuzzyRef.ref.id == pmPreRef.ref.id && pmFuzzyRef.ref.kind == pmPreRef.ref.kind && pmFuzzyRef.ref.module == pmPreRef.ref.module && pmFuzzyRef.ref.arch == pmPreRef.ref.arch && pmFuzzyRef.ref.channel == pmPreRef.ref.channel)
            {
                if (pmFuzzyRef.ref.kind == "app")
                {
                    if (KMVersion::parse(pmFuzzyRef.ref.version) > KMVersion::parse(pmPreRef.ref.version))
                    {
                        auto findFuzzyIt = std::find(d->m_pmFuzzyMatchRefs.begin(), d->m_pmFuzzyMatchRefs.end(), preRef);
                        if (findFuzzyIt != d->m_pmFuzzyMatchRefs.end())
                        {
                            it = d->m_pmFuzzyMatchRefs.erase(findFuzzyIt);
                        }
                        preRef = ref;
                        continue;
                    }
                }
                else
                {
                    std::istringstream currentVersion(pmFuzzyRef.ref.version);
                    std::istringstream preVersion(pmPreRef.ref.version);
                    int major, minor, patch, secure;
                    int preMajor, preMinor, prePatch, preSecure;
                    char dot;
                    currentVersion >> major >> dot >> minor >> dot >> patch;
                    preVersion >> preMajor >> dot >> preMinor >> dot >> prePatch;
                    if (major != preMajor || minor != preMinor)
                    {
                        preRef = ref;
                        continue;
                    }
                    else
                    {
                        if (patch >= prePatch)
                        {
                            auto findFuzzyIt = std::find(d->m_pmFuzzyMatchRefs.begin(), d->m_pmFuzzyMatchRefs.end(), preRef);
                            if (findFuzzyIt != d->m_pmFuzzyMatchRefs.end())
                            {
                                std::string idVersion = pmPreRef.ref.id + pmPreRef.ref.version;
                                auto idVersionIt = std::find(d->m_pmSpecialIdVersion.begin(), d->m_pmSpecialIdVersion.end(), idVersion);
                                if (idVersionIt == d->m_pmSpecialIdVersion.end())
                                {
                                    it = d->m_pmFuzzyMatchRefs.erase(findFuzzyIt);
                                }
                            }
                            preRef = ref;
                            continue;
                        }
                    }
                }
            }
            else
            {
                preRef = ref;
            }
        }
    }
    return;
}

void KMPackageManager::getDownloadRefList(TaskType taskType)
{
    for (auto &ref : d->m_pmFuzzyMatchRefs)
    {
        KMPMRef pmFuzzyRef = KMUtil::parseSummaryFuzzyKMPMRef(ref);
        switch (taskType)
        {
        case TASK_TYPE_INSTALL:
            // 安装时，过滤掉已安装的软件
            if (currentFuzzyRefIsInstalled(pmFuzzyRef))
            {
                std::string idVersion = pmFuzzyRef.ref.id + pmFuzzyRef.ref.version;
                auto it = std::find(d->m_pmSpecialIdVersion.begin(), d->m_pmSpecialIdVersion.end(), idVersion);
                if (it == d->m_pmSpecialIdVersion.end())
                {
                    auto idIt = std::find(d->m_pmInstallId.begin(), d->m_pmInstallId.end(), pmFuzzyRef.ref.id);

                    if (idIt != d->m_pmInstallId.end())
                    {
                        std::cout << _("This package has been installed! ") << pmFuzzyRef.ref.id << std::endl;
                    }

                    continue;
                }
            }
            break;
        case TASK_TYPE_UPDATE:
            // 更新时也是过滤掉已安装的软件
            if (currentFuzzyRefIsInstalled(pmFuzzyRef))
            {
                continue;
            }
            break;
        case TASK_TYPE_REPAIR:
            // 修复时不过滤掉已安装的软件
            if (!currentFuzzyRefIsInstalled(pmFuzzyRef))
            {
                KMPMRef pmRef = pmFuzzyRef;
                pmRef.ref.version = "";
                if (!currentRefIsInstalled(pmRef))
                {
                    continue;
                }
            }
            break;
        default:
            break;
        }

        kmlogger.debug("Installing ref : %s", ref.c_str());

        // 再次构建完整的 ostree ref
        pmFuzzyRef.completedRef = KMUtil::dupKMPMCompletedRef(pmFuzzyRef);
        // d->m_pmDownloadRefs.clear();
        d->m_pmDownloadRefs.push_back(pmFuzzyRef.completedRef);
        d->m_parseSummaryJson->getSummaryInfo(pmFuzzyRef);

        d->m_pmRefDownloadList.push_back(pmFuzzyRef);
        globalRefDownloadListForProgress.push_back(pmFuzzyRef);
    }
}

bool KMPackageManager::printUninstallDependsOnList(const vector<std::string> &UninstallVector)
{
    if (UninstallVector.empty())
    {
        return false;
    }

    d->m_formatter->setColumnWidth(35, 35, 15, 10, 25, 25, 30, 25, 35, 35);
    d->m_formatter->setShowArch(false);
    d->m_formatter->setShowSize(false);
    d->m_formatter->setShowName(false);
    d->m_formatter->setShowDescription(false);
    d->m_formatter->setShowChannel(true);
    d->m_formatter->printHeader();

    bool foundAnyEntries = false;
    bool printHeader = false;

    for (auto &uninstallIt : UninstallVector)
    {
        std::stringstream ss(uninstallIt);
        std::string token;
        vector<std::string> tokens;

        while (std::getline(ss, token, '/'))
        {
            tokens.push_back(token); // 将切分后的部分添加到 tokens 向量中
        }

        int tokensSize = tokens.size();
        KMEntry entry;
        entry.id = tokens[tokensSize - 3];
        entry.module = tokens[tokensSize - 2];
        entry.version = tokens[tokensSize - 1];
        entry.channel = tokens[tokensSize - 6];
        d->m_formatter->printEntry(entry);
    }

    if (!foundAnyEntries)
    {
        // std::cout << "No matching packages found for uninstallation.\n";
        return false;
    }

    d->m_formatter->printFooter();

    return foundAnyEntries;
}

void KMPackageManager::printDownloadRefList(const vector<KMPMRef> &downloadRefs)
{
    d->m_formatter->setColumnWidth(35, 35, 15, 10, 25, 25, 30, 25, 35, 35);
    d->m_formatter->setShowArch(false);
    d->m_formatter->setShowChannel(true);
    d->m_formatter->printHeader();

    for (const auto &pmRef : downloadRefs)
    {
        string installedSize = KMUtil::formatInstalledSize((double)pmRef.installedSize);

        KMEntry entry;
        entry.id = pmRef.ref.id;
        entry.name = pmRef.name;
        entry.description = pmRef.description;
        entry.module = pmRef.ref.module;
        entry.version = pmRef.ref.version;
        entry.arch = pmRef.ref.arch;
        entry.channel = pmRef.ref.channel;
        entry.size = installedSize;

        d->m_formatter->printEntry(entry);
    }

    d->m_formatter->printFooter();
}

bool KMPackageManager::printAssumeYesChoice(PackageManagerOptions pmOptions)
{
    if (!pmOptions.assumeYes)
    {
        std::cout << _("Do you want to install these changes? [Y/n]: ");

        char yesOption;

        while (true)
        {
            yesOption = std::cin.get();

            // 检查是否遇到 EOF(Ctrl+D)
            if (std::cin.eof())
            {
                std::cout << _("\nInput ended unexpectedly. Cancel install.\n");
                return false;
            }

            if (d->m_util->isYes(yesOption))
            {
                break;
            }
            else if (d->m_util->isNo(yesOption))
            {
                std::cout << _("Cancel install.\n");
                return false;
            }
            else
            {
                std::cout << _("Do you want to install these changes? [Y/n]: ");
            }
        }
    }

    return true;
}

int KMPackageManager::getTerminalWidth()
{
    struct winsize w;
    ioctl(STDOUT_FILENO, TIOCGWINSZ, &w);
    return w.ws_col;
}

/**
 * @brief 进度处理回调函数
 *
 * 进度条处理分成3个步骤：
 *  - 1. 当拉取 metadata 时， 增长到 5%;
 *  - 2. 下载数据一直到 97%;
 *  - 3. 写入 ostree objects 为最后的 3%
 *
 * @param progressStatus 在 ostree pull 时，发送的进度等信息结构体
 */
void KMPackageManager::progressCallback(ProgressStatus progressStatus)
{
    kmlogger.debug("enter installProgressCallback, current ref: %s", progressStatus.ref.c_str());
    char buf[1024 * 8];
    guint64 total = 0;
    guint64 elapsedTime;
    guint newProgress = 0;
    gboolean estimating = false;
    guint64 totalTransferred;
    gboolean lastWasMetadata = progressStatus.lastWasMetadata;
    gchar *formattedBytesTotalTransferred = NULL;

    int i;
    int fullSize, partial;
    char *speed = NULL;
    int barLength;
    const char *partialBlocks[] = {" ", "▏", "▎", "▍", "▌", "▋", "▊", "▉"};
    const char *fullBlock = "█";

    int rows = 24, cols = 80;

    elapsedTime = (g_get_monotonic_time() - progressStatus.startTime) / G_USEC_PER_SEC;
    kmlogger.debug("elapsedTime: %ld", elapsedTime);

    if (*progressStatus.ostreeStatus)
    {
        // 当ostree_status 有反馈，则代表fetch成功，总进度到了100%
        kmlogger.debug("install ostree status receive pull success");
        newProgress = 100;
        progressStatus.progress = 100;
    }
    else
    {
        totalTransferred = progressStatus.bytesTransferred + progressStatus.transferredExtraDataBytes;
        formattedBytesTotalTransferred = g_format_size_full(totalTransferred, G_FORMAT_SIZE_IEC_UNITS);

        progressStatus.lastWasMetadata = false;

        if (progressStatus.totalDeltaParts == 0 && (progressStatus.outstandingMetadataFetches > 0 || lastWasMetadata) && progressStatus.metadataFetched < 20)
        {
            /* 在没有完成 metadata 处理时，我们需要执行2次回调函数，
                    因为有时会遇到仅处理元数据会触发调用，但却没进行处理和添加更多元数据。*/
            if (progressStatus.outstandingMetadataFetches > 0)
            {
                progressStatus.lastWasMetadata = true;
            }

            /* At this point we don't really know how much data there is, so we have to make a
             * guess. Since its really hard to figure out early how much data there is we report 1%
             * until all objects are scanned. */

            estimating = true;

            sprintf(buf, _("Downloading metadata: %u/(estimating) %s"), progressStatus.fetched, formattedBytesTotalTransferred);
            kmlogger.debug("Downloading metadata buf: %s", buf);

            /* 增长到 5% 直到所有元数据都被拉取了 */
            newProgress = 0;
            if (progressStatus.requested > 0)
            {
                newProgress = progressStatus.fetched * 5 / progressStatus.requested;
            }
        }
        else
        {
            if (progressStatus.totalDeltaParts > 0)
            {
                gchar *formattedBytesTotal = NULL;

                /* 仅仅使用 deltas, 因此我们可以忽略 regular objects 来获得更好的 sizes;
                 *
                 * fetchedDeltaPartSize 所有 delta 部分 和 fallback objects 参数的总大小，
                 * 它是在最开始可以获得的，因此不用下载。
                 */
                total = progressStatus.totalDeltaPartSize - progressStatus.fetchedDeltaPartSize + progressStatus.totalExtraDataBytes;
                formattedBytesTotal = g_format_size_full(total, G_FORMAT_SIZE_IEC_UNITS);

                sprintf(buf, _("Downloading: %s/%s"), formattedBytesTotalTransferred, formattedBytesTotal);
                kmlogger.debug("Downloading buf: %s", buf);
            }
            else
            {
                /* Non-deltas, 因此我们不能知道除了 object 计数和额外数据的字节大小以外的任何东西，
                       为了比较它们和额外数据，我们使用平均对象大小来评估总大小。*/
                double averageObjectSize = 1;
                if (progressStatus.fetched > 0)
                {
                    averageObjectSize = progressStatus.bytesTransferred / (double)progressStatus.fetched;
                }

                total = averageObjectSize * progressStatus.requested + progressStatus.totalExtraDataBytes;

                gchar *formattedBytesTotal = g_format_size_full(total, G_FORMAT_SIZE_IEC_UNITS);
                sprintf(buf, _("Downloading extra data: %s/%s"), formattedBytesTotalTransferred, formattedBytesTotal);
                kmlogger.debug("下载额外数据 buf: %s", buf);

                sprintf(buf, _("Downloading files: %d/%d %s"), progressStatus.fetched, progressStatus.requested, formattedBytesTotalTransferred);
                kmlogger.debug("下载文件: %s", buf);
            }

            /* 下载总进度到达 97% */
            if (total > 0)
            {
                newProgress = 5 + ((totalTransferred / (gdouble)total) * 92); // 这个数据会因下载额外数据导致不准，比如会从 75 降到 48
            }
            else
            {
                newProgress = 97;
            }

            /* 增加最后写入对象的 3% 进度 */
            newProgress += getWriteProgress(progressStatus.outstandingWrites);
        }

        if (elapsedTime > 0) // Ignore first second
        {
            gchar *formattedBytesSec = g_format_size(totalTransferred / elapsedTime);
            sprintf(buf, " (%s/s)", formattedBytesSec);
            kmlogger.debug("totalTransferred: %ld elapsedTime: %ld buf: %s", totalTransferred, elapsedTime, buf);
        }
    }

    kmlogger.info("newProgress: %d currentPercent: %d", newProgress, currentPercent);
    // 当进度小于当前进度时，更新进度
    if (newProgress < currentPercent)
    {
        newProgress = currentPercent;
    }
    else
    {
        if (newProgress == 100)
        {
            currentPercent = 0;
        }
        else
        {
            currentPercent = newProgress;
        }
    }

    progressStatus.lastTotal = total;
    kmlogger.debug("newProgress: %d\n", newProgress);
    if (newProgress > 100)
    {
        if (!progressStatus.reportedOverflow)
        {
            kmlogger.error("Unexpectedly got > 100%% progress, limiting");
        }

        progressStatus.reportedOverflow = true;
        newProgress = 100;
    }

    guint singlePercent = newProgress; // 单一软件包下载进度，这个单一进度会因下载额外数据导致不准，比如会从 75 降到 48

    string currentRefId;

    uint64_t totalSize = getTotalSize();
    uint64_t singleSize = getSingleSize(progressStatus.ref, currentRefId);

    guint totalPercent = 0;    // 总进度
    guint currentProgress = 0; // 当前的进度
    string lastDownloadRef = globalRefDownloadListForProgress.back().completedRef;
    kmlogger.debug("lastDownloadRef: %s", lastDownloadRef.c_str());

    for (const auto &item : globalRefDownloadListForProgress) // 内部排序，因此不能顺序存储
    {
        kmlogger.debug("completedRef: %s size:%ld ", item.completedRef.c_str(), item.installedSize);
        // 当前下载的 ref 进度
        if (item.completedRef == progressStatus.ref)
        {
            currentProgress += (double)singlePercent / 100 * singleSize;
            break;
        }
        currentProgress += item.installedSize;
    }

    totalPercent = (double)currentProgress * 100 / totalSize;

    kmlogger.info("singlePercent:%d singleSize: %ld currentProgress: %ld  totalSize: %ld totalPercent: %d", singlePercent, singleSize, currentProgress, totalSize, totalPercent);

    progressStatus.progress = totalPercent;

    progressStatus.estimating = estimating;
    int speed_len = 0;

    if (elapsedTime > 0)
    {
        char *formattedBytesSec = g_format_size(totalTransferred / elapsedTime);
        string remaining;
        if (elapsedTime > 3 && singlePercent > 0)
        {
            guint64 totalTime = elapsedTime * 100 / (double)singlePercent;
            remaining = formatDuration(totalTime - elapsedTime);
            kmlogger.debug("totalTime:%ld elapsedTime:%ld  remaining_time:%ld remaining: %s", totalTime, elapsedTime, totalTime - elapsedTime, remaining.c_str());
        }

        guint64 remainingBytes = totalSize - currentProgress;
        string remainingStr = calculateDownloadTime(string(formattedBytesSec), remainingBytes);
        kmlogger.debug("remainingBytes: %ld remainingStr: %s preRemaining:%s", remainingBytes, remainingStr.c_str(), preRemaining.c_str());

        preRemaining = (remainingStr != "") ? remainingStr : preRemaining;
        // speed = g_strdup_printf("%s", formattedBytesSec);
        speed = g_strdup_printf("%s/s", formattedBytesSec);
        speed_len = MAX(speed_len, strlen(speed) + 2);
    }

    int terminalWidth = getTerminalWidth();
    barLength = terminalWidth - 25;
    fullSize = (barLength * totalPercent) / 100;
    partial = (((barLength * totalPercent) % 100) * G_N_ELEMENTS(partialBlocks)) / 100;

    if (terminalWidth != lastWidth)
    {
        lastWidth = terminalWidth;
    }

    char str[1024 * 8];

    // 构建静态信息部分
    std::string currentStaticInfo = "正在下载 " + currentRefId;

    // 只在静态信息变化或首次显示时输出
    if (currentStaticInfo != lastStaticInfo || !staticInfoDisplayed)
    {
        clearOldProgress();
        std::cout << currentStaticInfo << "\n";
        lastStaticInfo = currentStaticInfo;
        staticInfoDisplayed = true;
    }
    sprintf(str, "[");

    kmlogger.debug("fullSize: %d barLength:%d", fullSize, barLength);
    for (i = 0; i < barLength; i++)
    {
        if (i < fullSize)
        {
            strcat(str, "#");
        }
        else
        {
            strcat(str, ".");
        }
    }

    if (i < barLength)
    {
        sprintf(str + strlen(str), "%-20s", partialBlocks[partial]);
        kmlogger.debug("str: %s", str);
        i++;
    }

    sprintf(str + strlen(str), "] %3d%%", totalPercent);

    if (*progressStatus.ostreeStatus)
    {
        // printf("\n");

        if (elapsedTime > 0)
        {
            char *formattedBytesSec = g_format_size(totalTransferred / elapsedTime);
            speed = g_strdup_printf("%s/s", formattedBytesSec);

            guint64 remainingBytes = totalSize - currentProgress;
            string remainingStr = calculateDownloadTime(string(formattedBytesSec), remainingBytes).c_str();
            kmlogger.debug("remainingBytes: %ld remainingStr: %s preRemaining:%s", remainingBytes, remainingStr.c_str(), preRemaining.c_str());

            preRemaining = (remainingStr != "") ? remainingStr : preRemaining;
            speed = g_strdup_printf("%s/s", formattedBytesSec);
            speed_len = MAX(speed_len, strlen(speed) + 2);
        }
        else
        {
            // elapsedTime: 0
            kmlogger.debug("elapsedTime == 0");
            speed = g_strdup_printf("%s/s %s", "5 MB", "03:30");
            speed_len = MAX(speed_len, strlen(speed) + 2);
        }
        if (speed)
        {
            if (totalPercent < 100)
            {
                sprintf(str + strlen(str), " %s", speed);
            }
        }
    }
    else
    {
        if (speed)
        {
            if (totalPercent < 100)
            {
                sprintf(str + strlen(str), " %s", speed);
            }
        }
    }

    clearOldProgress();
    std::cout << str;

    lastOutput = string(str);
    lastProgressStr = lastOutput;

    kmlogger.debug("currentRef: %s lastRef: %s progress", progressStatus.ref.c_str(), progressStatus.lastRef.c_str());
    if (progressStatus.ref == progressStatus.lastRef && *progressStatus.ostreeStatus)
    {
        totalPercent = 100;
        printf("\n");
    }

    if (speed)
    {
        kmlogger.debug("speed: %s", speed);
    }
    else
    {
        speed = g_strdup_printf("5 MB/s");
        speed_len = MAX(speed_len, strlen(speed) + 2);
    }

    globalProgressStatus.speed = string(speed); // todo: 存在段错误
    globalProgressStatus.refId = currentRefId;
    globalProgressStatus.progress = totalPercent;
    globalProgressStatus.timeout = 5;

    kmlogger.debug("install progress------>>:  refId:%s progress:%d speed:%s timeout:%d", globalProgressStatus.refId.c_str(), globalProgressStatus.progress, globalProgressStatus.speed.c_str(), globalProgressStatus.timeout);

    progressJson = KMDbusJson::packProgressInfoToJson(globalProgressStatus.refId, globalProgressStatus.progress, globalProgressStatus.speed, globalProgressStatus.timeout);

    updateProgress();

    fflush(stdout); // 刷新输出缓冲区
}

void KMPackageManager::clearOldProgress()
{
    if (!lastProgressStr.empty() && lastWidth > 0)
    {
        int lines = (lastProgressStr.length() + lastWidth - 1) / lastWidth;
        std::cout << "\r";
        for (int i = 0; i < lines; ++i)
        {
            std::cout << "\033[K";
            if (i < lines - 1)
            {
                std::cout << "\033[A";
            }
        }
        std::cout << std::flush;
    }
}

int KMPackageManager::getWriteProgress(int outstandingWrites)
{
    return outstandingWrites > 0 ? (int)(3 / (double)outstandingWrites) : 3;
}

uint64_t KMPackageManager::getTotalSize()
{
    uint64_t sumSize = 0;
    for (const auto &ref : globalRefDownloadListForProgress)
    {
        kmlogger.debug("completedRef: %s installedSize:%ld", ref.completedRef.c_str(), ref.installedSize);

        sumSize += ref.installedSize;
    }

    return sumSize;
}

uint64_t KMPackageManager::getSingleSize(string completedRef, string &id)
{
    uint64_t singleSize;
    for (const auto &pmRef : globalRefDownloadListForProgress)
    {
        kmlogger.debug("completedRef: %s installedSize:%ld", pmRef.completedRef.c_str(), pmRef.installedSize);
        if (completedRef == pmRef.completedRef)
        {
            singleSize = pmRef.installedSize;
            id = pmRef.ref.id;
        }
    }

    return singleSize;
}

void KMPackageManager::updateProgress()
{
    // 只有当 globalProgressCallback
    // 是有效的时候才调用,即已经与某个可调用对象关联，并且没有被释放，避免 bad_function_call
    if (globalProgressCallback && progressJson != "")
    {
        globalProgressCallback(progressJson);
    }
}

string KMPackageManager::formatDuration(guint64 duration)
{
    int h, m, s;

    std::stringstream ss;

    m = duration / 60;
    s = duration % 60;
    h = m / 60;
    m = m % 60;

    if (h > 0)
    {
        // 使用流操作符进行格式化，输出为两位数
        ss.fill('0'); // 填充字符为 '0'
        ss.width(2);  // 设置宽度为 2
        ss << h << ":";
    }

    ss.fill('0');
    ss.width(2);
    ss << m << ":";

    ss.fill('0');
    ss.width(2);
    ss << s;

    return ss.str();
}

std::string KMPackageManager::calculateDownloadTime(const std::string &speed, guint64 sizeBytes)
{
    long int downloadSpeed;
    std::string result;

    kmlogger.debug("speed: %s sizeBytes: %ld", speed.c_str(), sizeBytes);

    // Parse the speed string to get the download speed in bytes/s
    if (speed.find("MB") != std::string::npos)
    {
        downloadSpeed = std::stol(speed.substr(0, speed.find("MB") - 1)) * 1024 * 1024; // Convert MB/s to bytes/s
    }
    else if (speed.find("kB") != std::string::npos)
    {
        downloadSpeed = std::stol(speed.substr(0, speed.find("kB") - 1)) * 1024; // Convert kB/s to bytes/s
    }
    else if (speed.find("bytes") != std::string::npos)
    {
        downloadSpeed = std::stol(speed.substr(0, speed.find("bytes") - 1));
        if (downloadSpeed == 0)
        {
            downloadSpeed = 5 * 1024; // 当 speed 属于 0 bytes/s 时，代表不下载，但进度也应该有，因此默认为5*1024
                                      // = 5KB 计算时间
        }
    }
    else
    {
        if (sizeBytes < 169887)
        {
            return "00:10"; // 给出默认时间 10 秒
        }
        else
        {
            return "01:00"; // 给出默认时间 1 分钟
        }
    }
    kmlogger.debug("speed: %s sizeBytes: %ld downloadSpeed=%ld", speed.c_str(), sizeBytes, downloadSpeed);

    // if (isZero(downloadSpeed))
    if (downloadSpeed == 0 && sizeBytes > 0)
    {
        // return "";
        downloadSpeed = 5 * 1024; // 当 speed 属于 0 bytes/s 时，代表不下载，但进度也应该有，因此默认为50计算时间
    }

    // Calculate the remaining download time in seconds
    int downloadTime = (int)sizeBytes / downloadSpeed;
    kmlogger.debug("speed: %s sizeBytes: %ld downloadSpeed=%ld downloadTime=%d", speed.c_str(), sizeBytes, downloadSpeed, downloadTime);

    // Format the download time as hours:minutes:seconds or minutes:seconds
    if (downloadTime >= 3600)
    {
        int hours = static_cast<int>(downloadTime / 3600);
        int minutes = static_cast<int>((downloadTime % 3600) / 60);
        int seconds = static_cast<int>(downloadTime % 60);
        result = std::to_string(hours) + ":" + std::to_string(minutes) + ":" + std::to_string(seconds);
        kmlogger.debug("speed: %s sizeBytes: %ld downloadSpeed=%ld downloadTime=%d result:%s", speed.c_str(), sizeBytes, downloadSpeed, downloadTime, result.c_str());
    }
    else
    {
        int minutes = static_cast<int>(downloadTime / 60);
        int seconds = static_cast<int>(downloadTime % 60);
        result = std::to_string(minutes) + ":" + std::to_string(seconds);
        kmlogger.debug("speed: %s sizeBytes: %ld downloadSpeed=%ld downloadTime=%d result:%s", speed.c_str(), sizeBytes, downloadSpeed, downloadTime, result.c_str());
    }

    if (result.find("-") != std::string::npos)
    {
        // 时间为负数不合理，此处先默认给出5分钟
        return "05:00";
    }

    return result;
}

int KMPackageManager::deploy()
{
    fs::path basePath = d->m_folder->getPath();
    fs::path layersPath;

    kmlogger.info("部署开始");
    for (auto &downloadRef : d->m_pmRefDownloadList)
    {
        d->m_installedFiles.clear();

        if (downloadRef.ref.module == "devel")
        {
            layersPath = fs::path(KMStorageDir::getUserBaseDirLocation());
        }
        else
        {
            layersPath = basePath / "layers";
        }

        kmlogger.info("completedRef: %s", downloadRef.completedRef.c_str());

        fs::path idPath = layersPath / downloadRef.ref.channel / downloadRef.ref.arch / downloadRef.ref.kind / downloadRef.ref.id;

        fs::path versionDataPath = layersPath / downloadRef.completedRef;
        kmlogger.debug("versionDataPath:  %s", versionDataPath.c_str());

        if (downloadRef.ref.kind == "app")
        {
            d->m_installedFiles.insert(idPath.string());
        }

        d->m_installedFiles.insert(versionDataPath.string());

        if (std::filesystem::exists(versionDataPath) || std::filesystem::create_directories(versionDataPath))
        {
            kmlogger.debug("Directory created or already exists successfully!");
        }

        // ostreeManager.checkout("f3b1314b51f2b8865089f13cbae5b7cd449b169f0b36530d27da74beb68aa492", "your-repo-path");
        kmlogger.debug("Begin to checkout ...");
        string latestCommit = d->m_ostreeHandler->readLatestCommit(downloadRef.completedRef);
        d->m_ostreeHandler->checkout(latestCommit, versionDataPath.string());
        removeHardlinksInPlace(versionDataPath);
        kmlogger.debug("End to checkout.");

        // add "active/files/.ref" => content ""
        // fs::path emptyFileRef = versionDataPath / "files/.ref";
        // createFileForEmptyString(emptyFileRef.string());

        kmlogger.info("baseType:  %s", downloadRef.ref.kind.c_str());
        d->m_hostAppDeploy->setHostAppConfig(versionDataPath, downloadRef.ref.id);
        if (d->m_hostAppDeploy->loadHostAppInfo())
        {
            d->m_isHostapp = true;
            d->m_hostAppDeploy->collectSynchronizedFiles();
            if (currentRefIsUninstalled(downloadRef))
            {
                KMInfo("ok already installed!");
            }
            else
            {
                KMInfo("ok not installed!");
            }

            try
            {
                d->m_hostAppDeploy->synchronizeFiles();
                KMInfo("Synchronization complete!");
            }
            catch (const std::exception &e)
            {
                KMError("Error during synchronization: " + std::string(e.what()));
            }
        }
        else if (downloadRef.ref.kind == "app")
        {
            // create current symbolic path
            fs::path completedIdPath = layersPath / downloadRef.completedRef;

            // modify service file
            fs::path serviceFileDirectoryPath = versionDataPath / "entries" / "usr" / "share" / "dbus-1" / "services";

            if (fs::exists(serviceFileDirectoryPath))
            {
                modifyServiceFile(downloadRef, serviceFileDirectoryPath.string());
            }
            else
            {
                fs::path serviceFileDirectoryPath = versionDataPath / "entries" / "share" / "dbus-1" / "services";
                modifyServiceFile(downloadRef, serviceFileDirectoryPath.string());
            }

            string userFilePath;
            string systemFilePath;

            if (KMFileUtils::isV11())
            {
                userFilePath = "/opt/system/lib/systemd/user/";
                systemFilePath = "/opt/system/lib/systemd/system/";
                kmlogger.info("V11");
            }
            else
            {
                userFilePath = "/usr/lib/systemd/user/";
                systemFilePath = "/usr/lib/systemd/system/";
                kmlogger.info("Not V11");
            }

            fs::path appServiceFileDirectoryPath =fs::exists(versionDataPath / "entries" / "share")? versionDataPath / "entries" / "share" / "usr" / "lib" / "systemd" / "system" : versionDataPath / "entries" / "usr" / "lib" / "systemd" / "system";
            std::vector<std::string> files;
            vector<string> serviceNames;

            if (fs::exists(appServiceFileDirectoryPath))
            {
                bool success = getAllFilesInDirectory(appServiceFileDirectoryPath, files);

                if (success)
                {
                    serviceNames = getServiceName(files);

                    if (!serviceNames.empty())
                    {
                        for (auto serviceName : serviceNames)
                        {
                            size_t lastSlashPos = serviceName.find_last_of('/');
                            if (lastSlashPos != std::string::npos)
                            {
                                std::string lastPart = serviceName.substr(lastSlashPos + 1);
                                std::string systemDbusFilePath = systemFilePath + lastPart;
                                modifyAppServiceFile(downloadRef, serviceName, systemDbusFilePath);
                                d->m_installedFiles.insert(systemDbusFilePath);
                            }
                        }
                    }
                }
            }
            else
            {
                appServiceFileDirectoryPath =fs::exists(versionDataPath / "entries" / "share")? versionDataPath / "entries" / "share" / "usr" / "lib" / "systemd" / "user" : versionDataPath / "entries" / "usr" / "lib" / "systemd" / "user";

                if (fs::exists(appServiceFileDirectoryPath))
                {
                    bool success = getAllFilesInDirectory(appServiceFileDirectoryPath, files);
                    if (success)
                    {
                        serviceNames = getServiceName(files);

                        if (!serviceNames.empty())
                        {
                            for (auto serviceName : serviceNames)
                            {
                                size_t lastSlashPos = serviceName.find_last_of('/');
                                if (lastSlashPos != std::string::npos)
                                {
                                    std::string lastPart = serviceName.substr(lastSlashPos + 1);
                                    std::string userDbusFilePath = userFilePath + lastPart;
                                    modifyAppServiceFile(downloadRef, serviceName, userDbusFilePath);
                                    d->m_installedFiles.insert(userDbusFilePath);
                                }
                            }
                        }
                    }
                }
            }

            // Create binData file: entries/bin/ref.id
            fs::path binData;

            if (fs::exists(versionDataPath / "entries" / "usr"))
            {
                binData = versionDataPath / "entries" / "usr" / "bin";
            }
            else
            {
                binData = versionDataPath / "entries" / "bin";
            }

            kmlogger.debug("Check directory:  %s", binData.string().c_str());
            if (fs::exists(binData) || fs::create_directories(binData))
            {
                kmlogger.info("create diretory %s success", binData.string().c_str());
            }

            fs::path binDataRef = binData / downloadRef.ref.id;
            createExportBinShellScript(downloadRef, binDataRef.string());

            // deploy data: move "app/ref.id/current/active/entries/" to "/var/opt/kaiming/"
            // fs::path entriessPath = basePath / "entriess"; // For immutable system, we sheild this
            fs::path basePath = d->m_folder->getPath();
            fs::path exportsPath = d->m_folder->getExportsPath();
            fs::path appEntriesPath = versionDataPath / "entries";
            fs::path exportsSharePath = d->m_folder->getExportsSharePath();
            fs::path exportsBinPath = d->m_folder->getExportsBinPath();

            // 创建目标目录
            if (!fs::exists(exportsPath))
            {
                fs::create_directory(exportsPath);
            }

            if (!fs::exists(exportsSharePath))
            {
                fs::create_directory(exportsSharePath);
            }

            if (!fs::exists(exportsBinPath))
            {
                fs::create_directory(exportsBinPath);
            }

            // 递归遍历源目录export/并处理文件
            modifyExportsDirectory(downloadRef, appEntriesPath.string(), exportsPath.string());
            modifyExportsDirectory(downloadRef, appEntriesPath.string(), basePath.string());
            kmlogger.debug("Exports files symlinked to /opt/kaiming/exports directory successfully.");

            // 生成应用desktop, icon, service等缓存
            std::vector<std::string> triggerCachePaths = {"/opt/kaiming-tools/share/kaiming/shells/gtk-icon-cache.trigger", "/opt/kaiming-tools/share/kaiming/shells/mime-database.trigger", "/opt/kaiming-tools/share/kaiming/shells/desktop-database.trigger"};

            runMultiperTriggerUpdateCacheCommand(triggerCachePaths);
        }

        // 修改文件夹属主为当前用户，处理 run 运行的权限问题
        modifyDirOwnership(idPath);

        // Todo: Pruning repo: ostree_repo_prune()

        // KMFileUtils::createFileIfNotExist(deployDataFile.string());

        // 生成所有文件列表文件
        std::cout << "\nWaiting for generate " << downloadRef.name << "'s deploy file ..."
                  << "\n";
        bool isFound;

        if (downloadRef.ref.kind == "app")
        {
            isFound = getAllAppFilesInDirectory(versionDataPath.string(), d->m_installedFiles, downloadRef.ref.id);
        }
        else
        {
            isFound = getAllBaseFilesInDirectory(versionDataPath.string(), d->m_installedFiles);
        }

        if (!isFound)
        {
            std::cout << "Get installed file failed!" << std::endl;
        }

        fs::path deployDataFile = versionDataPath / "deploy";
        bool isDeploy = createInstalledFile(deployDataFile.string());
        if (!isDeploy)
        {
            std::cout << "Generate installed deploy file failed!" << std::endl;
        }

        fs::path infoListFile = d->m_folder->getInfoPath() / (downloadRef.ref.id + ".list");
        writeInstalledFilesToList(infoListFile.string());

        std::set<std::string> deployfiles = d->m_hostAppDeploy->getSynchronizedFiles();
        d->m_installedFiles.merge(deployfiles);
        setKsafLabelAndExectl(d->m_installedFiles, downloadRef.ref.id, downloadRef.ref.kind, versionDataPath);

        if (downloadRef.ref.module == "devel")
        {
            modifyDirOwnership(KMStorageDir::getUserDevelDataDirLocation());
        }

        // 创建 deploy ref
        d->m_ostreeHandler->updateDeployRef(downloadRef.completedRef, latestCommit);
    }
    d->m_pmDownloadRefs.clear();
    d->m_pmFuzzyMatchRefs.clear();
    d->m_pmRefDownloadList.clear();
    kmlogger.info("部署成功");

    return KM_ERROR_NO;
}

void KMPackageManager::modifyDirOwnership(const fs::path &path)
{
    std::string loginUser = KMStorageDir::getLoginUser();
    kmlogger.info("Current login user: %s", loginUser.c_str());

    if (loginUser.empty())
    {
        return;
    }

    string ownerShip = loginUser + ":" + loginUser;

    // 更改用户属组为当前登录用户
    std::vector<std::string> args;
    args.push_back("-R");
    args.push_back(ownerShip);
    args.push_back(path.string());

    if (!KMProcessUtils::spawn("chown", args))
    {
        kmlogger.error("Failed to chown file:  %s ", path.string().c_str());
    }
}

int KMPackageManager::deployOfflinePackage(std::map<std::string, string> &mountMap)
{
    fs::path basePath = d->m_folder->getPath();
    fs::path layersPath;

    uint64_t totalSize = 0;
    uint64_t preSize = 0;

    //计算总的包大小
    for (auto &downloadRef : d->m_pmRefOfflineList)
    {
        totalSize += downloadRef.installedSize;
    }

    kmlogger.info("部署开始");
    for (auto &downloadRef : d->m_pmRefOfflineList)
    {
        std::cout << "正在部署离线包: " << downloadRef.ref.id << " ..." << std::endl;
        kmlogger.info("正在部署 completedRef: %s", downloadRef.completedRef.c_str());

        if (downloadRef.ref.module == "devel")
        {
            layersPath = fs::path(KMStorageDir::getUserBaseDirLocation());
        }
        else
        {
            layersPath = basePath / "layers";
        }

        fs::path idPath = layersPath / downloadRef.ref.channel / downloadRef.ref.arch / downloadRef.ref.kind / downloadRef.ref.id;

        fs::path versionDataPath = layersPath / downloadRef.completedRef;
        kmlogger.debug("versionDataPath:  %s", versionDataPath.c_str());
        d->m_installedFiles.clear();

        if (downloadRef.ref.kind == "app")
        {
            d->m_installedFiles.insert(idPath.string());
        }

        d->m_installedFiles.insert(versionDataPath.string());

        if (std::filesystem::exists(versionDataPath) || std::filesystem::create_directories(versionDataPath))
        {
            kmlogger.debug("Directory created or already exists successfully!");
        }

        kmlogger.debug("Begin to copy data from mountpoint ...");
        fs::path mountValue = mountMap[downloadRef.ref.id + downloadRef.ref.module];
        fs::path mountpointModulePath = mountValue / "layers" / downloadRef.ref.id / downloadRef.ref.module;
        if (std::filesystem::exists(mountpointModulePath))
        {
            std::string sourcePath = KMUtil::escapePath(mountpointModulePath);
            std::string DestPath = KMUtil::escapePath(versionDataPath);
            std::string command = "rsync -a --progress " + sourcePath + " " + DestPath;

            FILE *pipe = popen(command.c_str(), "r");
            if (!pipe)
            {
                std::cerr << "Failed to run rsync command!" << std::endl;
                return 1;
            }

            int printNumber = -1;
            sharedValue = 0;
            // // 启动线程来监控rsync的进度
            std::thread progressThread(KMPackageManager::monitorProgress, pipe);

            while (printNumber != 100)
            {
                if (downloadRef.installedSize > 0)
                {
                    int currentValue = sharedValue.load(std::memory_order_relaxed);
                    if (currentValue > printNumber)
                    {
                        printNumber = currentValue;
                        //当前百分比等于已安装的+正在安装*当前包所占百分比
                        // std::string printValue =std::to_string(preSize + (currentValue * downloadRef.installedSize / totalSize));
                        std::cout << "\r" << currentValue << "%" << std::flush;
                        progressJson = currentValue + "%";
                        updateProgress();
                    }
                }

                if (feof(pipe))
                {
                    std::cout << "\r"
                              << "100%" << std::flush;
                    break;
                }
                std::this_thread::sleep_for(std::chrono::milliseconds(500)); // 适当间隔去获取值，避免过于频繁占用资源
            }

            // 等待进度监控线程和进度条线程完成
            progressThread.join();

            // 关闭管道
            fclose(pipe);

            kmlogger.debug("Data copied from mountpoint successfully.");
        }
        else
        {
            kmlogger.error("mountpointModulePath not exists:  %s", mountpointModulePath.string().c_str());
        }

        kmlogger.debug("End to copy data from mountpoint.");

        // add "active/files/.ref" => content ""
        // fs::path emptyFileRef = versionDataPath / "files/.ref";
        // createFileForEmptyString(emptyFileRef.string());
        kmlogger.info("baseType:  %s", downloadRef.ref.kind.c_str());

        d->m_hostAppDeploy->setHostAppConfig(versionDataPath, downloadRef.ref.id);
        if (d->m_hostAppDeploy->loadHostAppInfo())
        {
            d->m_isHostapp = true;
            d->m_hostAppDeploy->collectSynchronizedFiles();
            if (currentRefIsUninstalled(downloadRef))
            {
                KMInfo("ok already installed!");
            }
            else
            {
                KMInfo("ok not installed!");
            }

            try
            {
                d->m_hostAppDeploy->synchronizeFiles();
                KMInfo("Synchronization complete!");
            }
            catch (const std::exception &e)
            {
                KMError("Error during synchronization: " + std::string(e.what()));
            }
        }
        else if (downloadRef.ref.kind == "app")
        {
            // create current symbolic path
            fs::path completedIdPath = layersPath / downloadRef.completedRef;

            // modify service file
            fs::path serviceFileDirectoryPath = versionDataPath / "entries" / "usr" / "share" / "dbus-1" / "services";

            if (fs::exists(serviceFileDirectoryPath))
            {
                modifyServiceFile(downloadRef, serviceFileDirectoryPath.string());
            }
            else
            {
                fs::path serviceFileDirectoryPath = versionDataPath / "entries" / "share" / "dbus-1" / "services";
                modifyServiceFile(downloadRef, serviceFileDirectoryPath.string());
            }

            string userFilePath;
            string systemFilePath;

            if (KMFileUtils::isV11())
            {
                userFilePath = "/opt/system/lib/systemd/user/";
                systemFilePath = "/opt/system/lib/systemd/system/";
                kmlogger.info("V11");
            }
            else
            {
                userFilePath = "/usr/lib/systemd/user/";
                systemFilePath = "/usr/lib/systemd/system/";
                kmlogger.info("Not V11");
            }

            fs::path appServiceFileDirectoryPath =fs::exists(versionDataPath / "entries" / "share")? versionDataPath / "entries" / "share" / "usr" / "lib" / "systemd" / "system" : versionDataPath / "entries" / "usr" / "lib" / "systemd" / "system";
            std::vector<std::string> files;
            vector<string> serviceNames;

            if (fs::exists(appServiceFileDirectoryPath))
            {
                bool success = getAllFilesInDirectory(appServiceFileDirectoryPath, files);

                if (success)
                {
                    serviceNames = getServiceName(files);
                    if (!serviceNames.empty())
                    {
                        for (auto serviceName : serviceNames)
                        {
                            size_t lastSlashPos = serviceName.find_last_of('/');
                            if (lastSlashPos != std::string::npos)
                            {
                                std::string lastPart = serviceName.substr(lastSlashPos + 1);
                                std::string systemDbusFilePath = systemFilePath + lastPart;
                                modifyAppServiceFile(downloadRef, serviceName, systemDbusFilePath);
                                d->m_installedFiles.insert(systemDbusFilePath);
                            }
                        }
                    }
                }
            }
            else
            {
                fs::path appServiceFileDirectoryPath =fs::exists(versionDataPath / "entries" / "share")? versionDataPath / "entries" / "share" / "usr" / "lib" / "systemd" / "user" : versionDataPath / "entries" / "usr" / "lib" / "systemd" / "user";

                if (fs::exists(appServiceFileDirectoryPath))
                {
                    bool success = getAllFilesInDirectory(appServiceFileDirectoryPath, files);
                    if (success)
                    {
                        serviceNames = getServiceName(files);

                        if (!serviceNames.empty())
                        {
                            for (auto serviceName : serviceNames)
                            {
                                size_t lastSlashPos = serviceName.find_last_of('/');
                                if (lastSlashPos != std::string::npos)
                                {
                                    std::string lastPart = serviceName.substr(lastSlashPos + 1);
                                    std::string userDbusFilePath = userFilePath + lastPart;
                                    modifyAppServiceFile(downloadRef, serviceName, userDbusFilePath);
                                    d->m_installedFiles.insert(userDbusFilePath);
                                }
                            }
                        }
                    }
                }
            }

            // Create binData file: entries/bin/ref.id
            fs::path binData;

            if (fs::exists(versionDataPath / "entries" / "usr"))
            {
                binData = versionDataPath / "entries" / "usr" / "bin";
            }
            else
            {
                binData = versionDataPath / "entries" / "bin";
            }
            kmlogger.debug("Check directory:  %s", binData.string().c_str());
            if (fs::exists(binData) || fs::create_directories(binData))
            {
                kmlogger.info("create diretory %s success", binData.string().c_str());
            }

            fs::path binDataRef = binData / downloadRef.ref.id;
            createExportBinShellScript(downloadRef, binDataRef.string());

            // deploy data: move "app/ref.id/current/active/entries/" to "/var/opt/kaiming/"
            // fs::path entriessPath = basePath / "entriess"; // For immutable system, we sheild this
            fs::path exportsPath = d->m_folder->getExportsPath();
            fs::path appEntriesPath = versionDataPath / "entries";
            fs::path exportsSharePath = d->m_folder->getExportsSharePath();
            fs::path exportsBinPath = d->m_folder->getExportsBinPath();

            // 创建目标目录
            if (!fs::exists(exportsPath))
            {
                fs::create_directory(exportsPath);
            }

            if (!fs::exists(exportsSharePath))
            {
                fs::create_directory(exportsSharePath);
            }

            if (!fs::exists(exportsBinPath))
            {
                fs::create_directory(exportsBinPath);
            }

            // 递归遍历源目录export/并处理文件
            modifyExportsDirectory(downloadRef, appEntriesPath.string(), exportsPath.string());
            modifyExportsDirectory(downloadRef, appEntriesPath.string(), basePath.string());
            kmlogger.debug("Exports files symlinked to /opt/kaiming/exports directory successfully.");

            // 生成应用desktop, icon, service等缓存
            std::vector<std::string> triggerCachePaths = {"/opt/kaiming-tools/share/kaiming/shells/gtk-icon-cache.trigger", "/opt/kaiming-tools/share/kaiming/shells/mime-database.trigger", "/opt/kaiming-tools/share/kaiming/shells/desktop-database.trigger"};

            runMultiperTriggerUpdateCacheCommand(triggerCachePaths);

            // KMFileUtils::createFileIfNotExist(deployDataFile.string());
        }

        // 修改文件夹属主为当前用户，处理 run 运行的权限问题
        modifyDirOwnership(idPath);

        // 生成所有文件列表文件
        std::cout << "\nWaiting for generate " << downloadRef.name << "'s deploy file ..."
                  << "\n";
        bool isFound;

        if (downloadRef.ref.kind == "app")
        {
            isFound = getAllAppFilesInDirectory(versionDataPath.string(), d->m_installedFiles, downloadRef.ref.id);
        }
        else
        {
            isFound = getAllBaseFilesInDirectory(versionDataPath.string(), d->m_installedFiles);
        }

        if (!isFound)
        {
            std::cout << "Get installed file failed!" << std::endl;
        }

        fs::path deployDataFile = versionDataPath / "deploy";
        bool isDeploy = createInstalledFile(deployDataFile.string());
        if (!isDeploy)
        {
            std::cout << "Generate installed deploy file failed!" << std::endl;
        }

        fs::path infoListFile = d->m_folder->getInfoPath() / (downloadRef.ref.id + ".list");
        writeInstalledFilesToList(infoListFile.string());

        std::set<std::string> deployfiles = d->m_hostAppDeploy->getSynchronizedFiles();
        d->m_installedFiles.merge(deployfiles);
        setKsafLabelAndExectl(d->m_installedFiles, downloadRef.ref.id, downloadRef.ref.kind, versionDataPath);

        if (downloadRef.ref.module == "devel")
        {
            modifyDirOwnership(KMStorageDir::getUserDevelDataDirLocation());
        }
        // Todo: Pruning repo: ostree_repo_prune()
        std::cout << "成功部署离线包: " << downloadRef.ref.id << " " << std::endl;
    }

    // std::cout << "100%" << std::endl;
    progressJson = "100%";
    updateProgress();
    d->m_okFile->umountOK();
    kmlogger.info("部署成功");

    return KM_ERROR_NO;
}

void KMPackageManager::createFileForEmptyString(const std::string &filename)
{
    // 打开文件进行检查
    std::ifstream inputFile(filename);
    if (inputFile.is_open())
    {
        kmlogger.error("File already exists:  %s", filename.c_str());
        inputFile.close();
        return;
    }

    // 创建文件并打开输出流
    std::ofstream outputFile(filename);
    if (!outputFile.is_open())
    {
        kmlogger.error("Failed to create file:  %s", filename.c_str());
        return;
    }

    // 写入空字符串到文件
    outputFile << "";

    // 关闭文件
    outputFile.close();
}

void KMPackageManager::modifyServiceFile(KMPMRef &pmRef, string serviceFileDirectoryPath)
{
    std::vector<std::string> files;
    files.clear();

    bool noModifyServiceFileFlag = false;

    // 调用函数获取目录下所有文件
    bool success = getAllFilesInDirectory(serviceFileDirectoryPath, files);
    if (success)
    {
        for (const auto &file : files)
        {
            if (file.find(".service") != std::string::npos)
            {
                string serviceFileName = file;
                kmlogger.debug("serviceFileName:  %s", serviceFileName.c_str());

                std::vector<std::string> serviceDatas = getDataFromKVFile(serviceFileName);

                for (auto &line : serviceDatas)
                {
                    // 修改 Exec 字段
                    size_t eqPos = line.find("Exec=");
                    if (eqPos != std::string::npos && eqPos == 0)
                    {
                        std::string value = line.substr(eqPos + strlen("Exec="));
                        if (value.find(" run") != std::string::npos)
                        {
                            // 第一次已进行修改，因此不修改当前文件
                            noModifyServiceFileFlag = true;
                        }

                        string serviceExec = "Exec=";
                        string content = serviceExec.append("/opt/kaiming-tools/bin/kaiming run ").append("--branch=").append(pmRef.ref.channel).append(" ").append("--arch=").append(pmRef.ref.arch).append(" ").append("--command=").append(value).append(" ").append(pmRef.ref.id);
                        line = content;
                    }
                }

                if (!noModifyServiceFileFlag)
                {
                    // 将修改后的内容写回 .service 文件
                    if (setDataToKVFile(serviceFileName, serviceDatas))
                    {
                        kmlogger.debug("Desktop file modified successfully.");
                    }
                }
            }
        }
    }
    else
    {
        kmlogger.error("Failed to get files in directory.");
    }
}

void KMPackageManager::initializeFixedModifications()
{
    fixedModifications = {
        {"ExecReload", [this](const auto &s, const auto &id)
         { return addOkPrefix(s, id); }},
        {"ExecStartPre", [this](const auto &s, const auto &id)
         { return addOkPrefix(s, id); }},
        {"ExecStart", [this](const auto &s, const auto &id)
         { return formatExecStart(s, id); }},
        {"Type", "simple"},
        {"CapabilityBoundingSet", "CAP_NET_BIND_SERVICE"},
        {"NoNewPrivileges", "false"},
        {"PrivateUsers", "false"},
        {"ProtectControlGroups", "false"},
        {"ProtectHostname", "false"},
        {"ProtectKernelLogs", "false"},
        {"ProtectProc", "invisible"},
        {"RestrictAddressFamilies", ""},
        {"RestrictNamespaces", "false"},
        {"SystemCallArchitectrues", "native"},
        {"SystemCallFilter", ""},
        {"NoExecPaths", ""},
        {"TasksMax", [this](const auto &s, const auto &id)
         { return changeTasks(s); }},
        {"KillMode", "control-group"}};
}

vector<string> KMPackageManager::getServiceName(const vector<string> &files)
{
    vector<string> servicesFiles;
    for (const auto &file : files)
    {
        if (file.find(".service") != std::string::npos)
        {
            kmlogger.debug("serviceFileName:  %s", file.c_str());
            servicesFiles.push_back(file);
        }
    }
    return servicesFiles;
}

string KMPackageManager::addOkPrefix(const string &original, const std::string id)
{
    return "/opt/kaiming-tools/bin/kaiming run --command=\"" + original + "\" " + id;
}

string KMPackageManager::addKillPrefix(const string& original,const std::string id)
{
    return "/opt/kaiming-tools/bin/kaiming kill " + id;
}

string KMPackageManager::changeTasks(const string &original)
{
    if (original == "infinity")
    {
        return "512";
    }
    return original;
}

string KMPackageManager::formatExecStart(const std::string &line, const std::string id)
{
    std::string value = line;
    if (!value.empty() && value.front() == '"')
        value = value.substr(1);
    if (!value.empty() && value.back() == '"')
        value.pop_back();

    // 保留 `--supervised systemd --daemonize no`（如果存在）
    std::string extraArgs;
    size_t supervisedPos = value.find("--supervised systemd");
    size_t daemonizePos = value.find("--daemonize no");
    if (supervisedPos != std::string::npos && daemonizePos != std::string::npos)
    {
        extraArgs = " --supervised systemd --daemonize no";
        // 移除 "--supervised systemd" 和 "--daemonize no"
        value.erase(supervisedPos, std::string("--supervised systemd").length());
        // 移除可能出现的多余空格
        while (supervisedPos < value.length() && value[supervisedPos] == ' ')
        {
            value.erase(supervisedPos, 1);
        }
        daemonizePos = value.find("--daemonize no");
        if (daemonizePos != std::string::npos)
        {
            value.erase(daemonizePos, std::string("--daemonize no").length());
            // 移除可能出现的多余空格
            while (daemonizePos < value.length() && (value[daemonizePos] == ' ' || value[daemonizePos - 1] == ' '))
            {
                value.erase(daemonizePos, 1);
            }
        }
    }

    return "/opt/kaiming-tools/bin/kaiming run --detach=false --command=\"" + value + "\" " + id + " " + extraArgs;
}

bool KMPackageManager::isField(const string &line, const string &field)
{
    return line.rfind(field + "=", 0) == 0;
}

string KMPackageManager::modifyFixedField(const string &line, const string id)
{
    static const std::unordered_set<std::string> commentFileds = {
        "CapabilityBoundingSet",
        "ProtectKernelTunables",
        "ProtectSystem",
        "User",
        "Group",
        "ExecStop",
        "ProtectHome",
        "ProtectSystem",
        "ProtectKernelTunables",
        "ExecPaths"
    };

    for (const auto& field : commentFileds)
    {
        if (isField(line, field)) {
            return "#" + line;  // 直接注释掉
        }
    }

    for (const auto &[field, modifier] : fixedModifications)
    {
        if (isField(line, field))
        {
            size_t equalPos = line.find('=');
            string originalValue = (equalPos != string::npos) ? line.substr(equalPos + 1) : "";

            if (holds_alternative<CallbackFunc>(modifier))
            {
                // 如果是回调函数，调用它
                CallbackFunc callback = get<CallbackFunc>(modifier);
                string newValue = callback(originalValue, id);
                return field + "=" + newValue;
            }
            else
            {
                // 如果是固定值
                string newValue = get<string>(modifier);
                return field + "=" + newValue;
            }
        }
    }
    return line; // 不修改
}

void KMPackageManager::modifyAppServiceFile(KMPMRef &pmRef, string serviceFile, string destPath)
{
    bool noModifyServiceFileFlag = false;

    std::vector<std::string> serviceDatas = getDataFromKVFile(serviceFile);

    bool inServiceSection = false;
    bool hasUser = false;
    bool hasGroup = false;
    size_t serviceSectionEnd = 0;

    for (size_t i = 0; i < serviceDatas.size(); ++i)
    {
        auto &currentLine = serviceDatas[i];
        if (currentLine.empty() || currentLine[0] == '#')
        {
            continue;
        }

        // 检查是否进入[Service]部分
        if (currentLine == "[Service]")
        {
            inServiceSection = true;
            serviceSectionEnd = i + 1;
            continue;
        }

        // 只在[Service]部分中修改
        if (inServiceSection)
        {
            string modifiedLine = modifyFixedField(currentLine, pmRef.ref.id);
            if (modifiedLine != currentLine)
            {
                currentLine = modifiedLine;
            }
        }
    }

    if (inServiceSection)
    {
        // 直接在服务段末尾插入这三个字段
        serviceDatas.insert(serviceDatas.begin() + serviceSectionEnd, "User=root");
        serviceDatas.insert(serviceDatas.begin() + serviceSectionEnd + 1, "Group=root");
        serviceDatas.insert(serviceDatas.begin() + serviceSectionEnd + 2, "ExecStop=" + addKillPrefix("", pmRef.ref.id));
        
        // 更新服务段结束位置（增加了3行）
        serviceSectionEnd += 3;
    }

    fs::path dest = destPath;
    
    if (!fs::exists(dest.parent_path()) )
    {
        if (fs::create_directories(dest.parent_path()))
        {
            if (setDataToKVFile(dest, serviceDatas))
            {
                kmlogger.debug("Service file modified successfully.");
            }
            else
            {
                kmlogger.error("Failed to get files in directory.");
            }
        }
    }
    else
    {
        if (setDataToKVFile(dest, serviceDatas))
        {
            kmlogger.debug("Service file modified successfully.");
        }
        else
        {
            kmlogger.error("Failed to get files in directory.");
        }
    }
}

bool KMPackageManager::getAllFilesInDirectory(const std::string &directoryPath, std::vector<std::string> &files)
{
    try
    {
        // 遍历目录中的文件
        for (const auto &entry : fs::recursive_directory_iterator(directoryPath, fs::directory_options::skip_permission_denied | fs::directory_options::follow_directory_symlink))
        {
            if (fs::is_symlink(entry.path()) && fs::read_symlink(entry.path()) == entry.path())
            {
                continue;
            }

            // 将文件路径添加到文件列表中
            files.push_back(entry.path().string());
        }

        return true; // 成功获取文件列表
    }
    catch (const std::exception &e)
    {
        kmlogger.error("Failed to get files: %s", e.what());
        return false; // 获取文件列表失败
    }
}

bool KMPackageManager::getAllAppFilesInDirectory(const std::string &directoryPath, std::set<std::string> &files, const std::string &id)
{
    static const std::vector<std::string> binDirs = {
        "files/bin", "files/local/bin", "files/usr/bin", "files/usr/local/bin", "files/opt"};

    KMInfoJson infoJson;
    const std::string jsonPath = directoryPath + "/" + "info.json";
    infoJson.loadFile(jsonPath);
    bool isSysapp = infoJson.getAnnotations().sysapp;

    const std::string infoDir = d->m_folder->getInfoPath();
    fs::create_directories(infoDir);

    std::vector<std::pair<std::string, std::string>> vecExecRecords;
    const fs::path dirPath(directoryPath);
    const std::string appPrefix = "/opt/apps/" + id + "/files";

    try
    {
        std::vector<fs::path> fullBinPaths;
        for (const auto& subdir : binDirs) {
            fullBinPaths.push_back(dirPath / subdir);
        }

        // 遍历目录中的文件
        for (const auto &entry : fs::recursive_directory_iterator(dirPath, fs::directory_options::skip_permission_denied | fs::directory_options::follow_directory_symlink))
        {
            const auto& path = entry.path();
            const auto pathStr = path.string();

            if (fs::is_symlink(entry.path()) && fs::read_symlink(entry.path()) == entry.path())
            {
                continue;
            }

            files.insert(pathStr);

            if (!KMFileUtils::isExecutableFile(path)) {
                continue;
            }

            for (const auto& binPath : fullBinPaths) {
                if (path.string().find(binPath.string()) == 0) {
                    const std::string execName = path.filename().string();
                    const fs::path linkPath = fs::path(KAIMING_BIN_PATH) / execName;

                    if (fs::exists(linkPath)) {
                        fs::remove(linkPath);
                    }
                    fs::create_symlink(KAIMING_PROGRESS_FULL_PATH, linkPath);

                    std::string relPath;
                    const auto pos = pathStr.find("files");
                    relPath = (pos != std::string::npos) ? pathStr.substr(pos + 5) : path.filename().string();
                    if (!isSysapp) {
                        relPath = appPrefix + relPath;
                    }
                    vecExecRecords.emplace_back(std::move(relPath), id);
                    break;
                }
            }
        }

        const std::string infoFile = (fs::path(infoDir) / (id + ".bin")).string();
        std::ofstream ofs(infoFile, std::ios::out | std::ios::trunc);
        if (ofs)
        {
            for (const auto &kv : vecExecRecords)
            {
                ofs << kv.first << " " << kv.second << "\n";
            }
        }

        return true; // 成功获取文件列表
    }
    catch (const std::exception &e)
    {
        kmlogger.error("Failed to get files: %s", e.what());
        return false; // 获取文件列表失败
    }
}

bool KMPackageManager::getAllBaseFilesInDirectory(const std::string &directoryPath, std::set<std::string> &files)
{
    try
    {
        // 遍历目录中的文件
        for (const auto &entry : fs::recursive_directory_iterator(directoryPath, fs::directory_options::skip_permission_denied))
        {
            if (fs::is_symlink(entry.path()) && fs::read_symlink(entry.path()) == entry.path())
            {
                continue;
            }

            // 将文件路径添加到文件列表中
            files.insert(entry.path().string());
        }

        return true; // 成功获取文件列表
    }
    catch (const std::exception &e)
    {
        kmlogger.error("Failed to get files: %s", e.what());
        return false; // 获取文件列表失败
    }
}

std::vector<std::string> KMPackageManager::getDataFromKVFile(const std::string &filename)
{
    std::vector<std::string> lines;
    std::ifstream file(filename);
    if (file.is_open())
    {
        std::string line;
        while (std::getline(file, line))
        {
            lines.push_back(line);
        }
        file.close();
    }
    else
    {
        kmlogger.error("Failed to open file::  %s", filename.c_str());
    }
    return lines;
}

bool KMPackageManager::setDataToKVFile(const std::string &filename, const std::vector<std::string> &newLines)
{
    std::ofstream file(filename);
    if (file.is_open())
    {
        for (const auto &line : newLines)
        {
            file << line << std::endl;
        }
        file.close();
        return true;
    }
    else
    {
        kmlogger.error("inFailed to open file for writingdex:  %s", filename.c_str());
        return false;
    }
}

void KMPackageManager::modifyDesktopFile(KMPMRef &pmRef, string exportDesktopFilePath)
{
    kmlogger.info("Begin to exportDesktopFilePath: %s\n", exportDesktopFilePath.c_str());
    std::vector<std::string> desktopDatas = getDataFromKVFile(exportDesktopFilePath);

    bool needAddKaimingTag = false;

    // 遍历 desktopDatas 通过每行 line 修改需要的字段
    for (auto &line : desktopDatas)
    {
        // 跳过开头为 # 的注释字段
        if (line[0] == '#')
        {
            continue;
        }

        // 注释以下内容处理无图标或开始菜单无法点击运行问题
        std::vector<std::string> annotationVector = {
            "DBusActivatable", // 注释 DBusActivatable 字段，否则org.gnome.Sudoku包任务栏无图标
            "StartupNotify",   // 注释 StartupNotify 字段，否则org.gnome.Aisleriot包任务栏无图标
            "TryExec"          // 注释 TryExec 字段，否则org.gnome.Gitg和org.ggsoft.etw在开始菜单无法点击运行
        };

        bool needAnnotation = false;
        for (auto it = annotationVector.begin(); it != annotationVector.end(); ++it)
        {
            if (line.find(*it) == 0)
            {
                needAnnotation = true;
                break;
            }
        }

        if (needAnnotation)
        {
            line = "# " + line;
            continue;
        }

        // 修改 Exec 字段，否则开始菜单应用运行不起来
        size_t eqPos = line.find("Exec=");
        if (eqPos != std::string::npos && eqPos == 0)
        {
            std::string value = line.substr(eqPos + strlen("Exec="));
            if (value.find(" run") != std::string::npos)
            {
                // 第一次已进行修改，或者代表已经有可用 /usr/bin/kaiming run，因此不修改。案例有在线源中的 org.gnome.baobab 包。
                size_t pos = line.find("/usr/bin/kaiming");

                if (pos != std::string::npos) {
                    line.replace(pos, 16, "/opt/kaiming-tools/bin/kaiming");
                }

                continue;
            }

            // formatExec correct: --command=baobab org.gnome.baobab %U
            string formatExec = d->m_util->formatExecString(value, pmRef.ref.id);
            kmlogger.info("formatExec: %s", formatExec.c_str());

            // kigo %u -qwindowicon kigo -qwindowtitle %c => kigo --file-forwarding org.kde.kigo @@u
            // %u @@ -qwindowicon kigo -qwindowtitle %c
            string serviceExec = "Exec=";
            string content = serviceExec.append("/opt/kaiming-tools/bin/kaiming run ").append(formatExec);
            line = content;
        }

        // 保留，暂不修改 Icon 字段
        if (line.find("Icon=") != std::string::npos)
        {
            size_t iconPos = line.find("=") + 1;
            std::string path = line.substr(iconPos);

            if (path.find("/usr") == 0) {
                path.replace(0, 4, "/opt/kaiming");
            }

            line = "Icon=" + path;
        }

        // 新增 X-Kaiming 字段，用于SDK，开始菜单等获取软件包名
        if (line.find("X-Kaiming=") != std::string::npos)
        {
            needAddKaimingTag = false;
        }
        else
        {
            needAddKaimingTag = true;
        }
    }

    if (needAddKaimingTag)
    {
        for (auto it = desktopDatas.begin(); it != desktopDatas.end(); ++it)
        {
            if (it->find("[Desktop Entry]") != std::string::npos)
            {
                string kaimingTag = "X-Kaiming=" + pmRef.ref.id;
                desktopDatas.insert(it + 1, kaimingTag);
                break;
            }
        }
    }

    // 将修改后的内容写回 .desktop 文件
    if (!setDataToKVFile(exportDesktopFilePath, desktopDatas))
    {
        kmlogger.error("Desktop file modified failed.");
    }
    return;
}

bool KMPackageManager::createExportBinShellScript(KMPMRef pmRef, string fileName)
{
    // 设置内容
    string shellHead = "#!/bin/sh";
    string content = shellHead.append("\nexec ").append("/opt/kaiming-tools/bin/kaiming run --branch=").append(pmRef.ref.channel).append(" --arch=").append(pmRef.ref.arch).append(" ").append(pmRef.ref.id).append(" \"$@\"\n");

    kmlogger.info("export bin bash content:\n%s", content.c_str());
    // 创建文件并写入内容
    std::ofstream file(fileName);
    if (file.is_open())
    {
        file << content;
        file.close();
    }
    else
    {
        kmlogger.error("Failed to open file:  %s", fileName.c_str());
        return false;
    }

    // 设置文件权限为 0755
    if (chmod(fileName.c_str(), 0755) != 0)
    {
        kmlogger.error("Failed to set file permission for:  %s", fileName.c_str());
        return false;
    }

    return true;
}

bool KMPackageManager::createInstalledFile(string fileName)
{
    // 设置内容
    string content;

    for (auto &installedFile : d->m_installedFiles)
    {
        content = content.append(installedFile).append("\n");
    }

    // fix: runtime, base多版本安装时，已有版本deploy 会修改当前安装版本
    if (fs::exists(fileName))
    {
        fs::remove(fileName);
    }

    // 创建文件并写入内容
    std::ofstream file(fileName);
    if (file.is_open())
    {
        file << content;
        file.close();
    }
    else
    {
        kmlogger.error("Failed to open file:  %s", fileName.c_str());
        return false;
    }

    // 设置文件权限为 0755
    if (chmod(fileName.c_str(), 0755) != 0)
    {
        kmlogger.error("Failed to set file permission for:  %s", fileName.c_str());
        return false;
    }

    return true;
}

void KMPackageManager::writeInstalledFilesToList(const std::string &filePath)
{
    fs::path dirPath = fs::path(filePath).parent_path();

    if (!fs::exists(dirPath))
    {
        if (!fs::create_directories(dirPath))
        {
            KMError("Error creating directory: " + dirPath.string());
            return;
        }
    }

    std::ofstream outFile(filePath);

    if (!outFile)
    {
        KMError("Error opening file: " + filePath);
        return;
    }

    for (const auto &installedFile : d->m_installedFiles)
    {
        outFile << installedFile << "\n";
    }

    outFile.close();
}

void KMPackageManager::modifyExportsDirectory(KMPMRef &pmRef, const std::string &srcDir, const std::string &destDir)
{
    for (const auto &entry : fs::directory_iterator(srcDir))
    {
        if (fs::is_directory(entry))
        {
            // 递归处理子目录
            std::string subSrcDir = entry.path().string();
            std::string subDestDir = destDir + "/" + entry.path().filename().string();

            string removeStr = "/usr";
            size_t pos = subDestDir.find(removeStr);
            if (pos != std::string::npos)
            {
                subDestDir.erase(pos, removeStr.length());
            }

            fs::create_directory(subDestDir);
            modifyExportsDirectory(pmRef, subSrcDir, subDestDir);
        }
        else
        {
            // 处理文件
            std::string srcFilePath = entry.path().string();
            std::string destFilePath = destDir + "/" + entry.path().filename().string();
            kmlogger.debug("srcFilePath:%s destFilePath:%s", srcFilePath.c_str(), destFilePath.c_str());

            string removeStr = "/usr";
            size_t pos = destFilePath.find(removeStr);
            if (pos != std::string::npos)
            {
                destFilePath.erase(pos, removeStr.length());
            }
            kmlogger.debug("srcFilePath:%s destFilePath:%s", srcFilePath.c_str(), destFilePath.c_str());

            if (fs::exists(fs::path(destFilePath)) && fs::is_symlink(fs::path(destFilePath)))
            {
                kmlogger.info("destFilePath  %s exists.", destFilePath.c_str());
                d->m_installedFiles.insert(destFilePath);
                continue;
            }

            if (srcFilePath.find(".desktop") != std::string::npos || srcFilePath.find(".service") != std::string::npos || srcFilePath.find(".svg") != std::string::npos || srcFilePath.find(".png") != std::string::npos)
            {

                // 如果是 .desktop, .service, .svg 文件，则创建相对路径软链接到目标目录
                // 此处以 ".desktop" 为后缀，否则会出现 "org.freedesktop.impl.portal.desktop.gtk.service" 也会进行修改。
                if (KMStringUtils::endsWith(srcFilePath, ".desktop"))
                {
                    modifyDesktopFile(pmRef, srcFilePath);
                }

                // 获取 srcFilePath 相对于 destFilePath 的相对路径
                std::string relativePath = fs::relative(fs::path(srcFilePath), fs::path(destFilePath).parent_path()).string();
                kmlogger.info("Export .desktop, .service, .svg, .png relativePath:  %s destFilePath: %s", relativePath.c_str(), destFilePath.c_str());

                if (fs::is_symlink(fs::path(destFilePath)) || fs::exists(fs::path(destFilePath)))
                {
                    // 如果存在，删除该文件或链接
                    kmlogger.info("destFilePath  %s exists. delete it.", destFilePath.c_str());
                    fs::remove(fs::path(destFilePath));
                }

                fs::create_symlink(fs::path(relativePath), fs::path(destFilePath));
                d->m_installedFiles.insert(destFilePath);
            }
            else if (srcFilePath.find("entries/usr/bin") != std::string::npos || srcFilePath.find("entries/bin") != std::string::npos)
            {
                // 如果是 entries/bin 二进制文件，则创建相对路径软链接到目标目录
                std::string relativePath = fs::relative(fs::path(srcFilePath), fs::path(destFilePath).parent_path()).string();
                kmlogger.info("Export bin relativePath:  %s destFilePath: %s", relativePath.c_str(), destFilePath.c_str());

                // 使用 fs::exists 检测软链接的存在，它会返回目标文件的存在状态，而不是软链接本身。因此，使用 fs::is_symlink 是判断软链接是否存在的正确方法。
                if (fs::is_symlink(fs::path(destFilePath)))
                {
                    // 如果存在，删除该文件或链接
                    kmlogger.info("destFilePath  %s exists. delete it.", destFilePath.c_str());
                    fs::remove(fs::path(destFilePath));
                }

                fs::create_symlink(fs::path(relativePath), fs::path(destFilePath));
                d->m_installedFiles.insert(destFilePath);
            }
        }
    }
}

void KMPackageManager::runMultiperTriggerUpdateCacheCommand(const vector<string> &cacheShellPaths)
{
    for (const auto &path : cacheShellPaths)
    {
        runTriggerUpdateCacheCommand(path);
    }
}

void KMPackageManager::runTriggerUpdateCacheCommand(const std::string &cacheShellPath)
{
    // 构建命令字符串，将脚本路径和参数拼接在一起
    std::string command = cacheShellPath + " " + "/opt/kaiming";

    kmlogger.info("Generate cache command: %s", command.c_str());

    // 执行命令
    int result = std::system(command.c_str());
    if (result != 0)
    {
        kmlogger.error("Failed to execute command:  %s", command.c_str());
    }
}

// 线程函数：实时读取rsync的输出并解析进度信息
void KMPackageManager::monitorProgress(FILE *pipe)
{
    char buffer[256];
    while (fgets(buffer, sizeof(buffer), pipe) != nullptr)
    {
        std::string line(buffer);
        // 查找包含百分比的行，表示文件传输的进度
        size_t firstPos = line.find('=');
        if (firstPos != std::string::npos)
        {
            size_t secondPos = line.find(')', firstPos + 1);
            // 提取进度百分比
            std::string progresStr = line.substr(firstPos, secondPos); // 获取进度
            size_t thirdPos = progresStr.find('/');
            std::string numBeforeStr = progresStr.substr(1, thirdPos - 1);
            int numBefore, numAfter;
            std::stringstream(numBeforeStr) >> numBefore;
            std::string numAfterStr = progresStr.substr(thirdPos + 1, progresStr.length() - thirdPos - 3);
            std::stringstream(numAfterStr) >> numAfter;
            int result = (numAfter - numBefore);
            int divid = (int)((double)result / numAfter * 100);
            sharedValue.store(divid, std::memory_order_relaxed); // 在线程中修改共享变量的值
        }
    }
}

int KMPackageManager::verifySignature(std::string id, std::string kind, fs::path path)
{
    int verifyResult = KM_ERROR_NO;

    // 判断是否是 base 或 runtime，若是，则暂时不需要验签，直接安装
    if (kind == "base" || kind == "runtime")
    {
        return KM_ERROR_NO;
    }

    // 获取安全模式
    int safeStatus = getSafeMode();

    if (safeStatus == 1)
    {
        kmlogger.info("安全模式为关闭模式，跳过验签");
        return KM_ERROR_NO;
    }

    // 判断是否存在签名工具
    bool isSignTool = isKylinFileSignToolPresent();
    if (isSignTool)
    {
        kmlogger.info("kylin-signtool exists.");
        std::string sigFilesPath = (path / "sig").string();
        std::string filesMd5 = KMStringUtils::buildFilename(path, "files.md5");
        std::string filesSign = KMStringUtils::buildFilename(sigFilesPath, "files.sig");
        std::string filesPem = KMStringUtils::buildFilename(sigFilesPath, "files.pem");

        // 签名工具存在，则判断签名文件.md5是否存在
        if (KMFileUtils::fileExists(filesMd5))
        {
            kmlogger.info("files.md5 exists.");
            kmlogger.info("开始验签");
            std::string command = "kylinfilesigntool -v \"" + filesMd5 + "\" -s \"" + filesSign + "\" -c \"" + filesPem + "\"";
            verifyResult = std::system(command.c_str());
            if (verifyResult != KM_ERROR_NO)
            {
                kmlogger.info("验签失败");
                std::cout << _("Signature verification failed") << std::endl;
                if (safeStatus != 0)
                {
                    int errorCode = resolveFailedVerification(safeStatus, path, id);
                    if (errorCode != KM_ERROR_NO)
                    {
                        return errorCode;
                    }
                }
            }
            else
            {
                kmlogger.info("验签成功");

                if (!KMFileUtils::isObsolete(filesMd5))
                {
                    kmlogger.info("MD5 file is not old");
                    KMDirMd5sum md5sum(path);
                    if (!md5sum.checksum("files"))
                    {
                        kmlogger.info("cheksum MD5 files, the result is KM_ERROR_MD5_CHECK_FAILED");
                        return KM_ERROR_MD5_CHECK_FAILED;
                    }
                    else
                    {
                        kmlogger.info("cheksum MD5 files, the result is success");
                    }
                }
            }
        }
        else
        {
            kmlogger.info("签名文件不存在，停止安装");
            std::cout << _("Signature file not exist, installation stopped") << std::endl;
            return KM_ERROR_NO_MD5_FILES;
        }
    }
    else
    {
        kmlogger.info("签名工具不存在，不进行验签");
    }

    return KM_ERROR_NO;
}

// 判断签名工具是否存在
bool KMPackageManager::isKylinFileSignToolPresent()
{
    int result = std::system("which kylinfilesigntool > /dev/null 2>&1");
    // 结果为0，找到工具
    return (result == 0);
}

KMErrorCodeEnum KMPackageManager::setKsafLabelAndExectl(const std::set<std::string> &files, const std::string &appName, const std::string &kind, const std::string &path)
{
    if (files.empty() || appName.empty())
    {
        kmlogger.info("Invalid parameters: files or appName is empty");
        return KM_ERROR_INVALID_PARAM;
    }

    KMLibHandle ksafLabelHandle("/usr/lib/libksaflabelmgr.so");
    KMLibHandle exectlHandle("/usr/lib/libkysec2_exectl.so.0");
    if (!ksafLabelHandle || !exectlHandle)
    {
        kmlogger.info("Failed to load required libraries");
        return KM_ERROR_SETLABEL_FAILED;
    }

    using SetLabelFunc = int (*)(const char **, const char *);
    using XattrSetFunc = int (*)(char *[], size_t);
    using SetLabelForInterpreterFunc = int (*)(const char *);

    std::vector<const char *> c_files;
    std::vector<char *> c_paths;
    c_files.reserve(files.size() + 1);
    c_paths.reserve(files.size());
    for (const auto &file : files)
    {
        c_files.push_back(file.c_str());
        c_paths.push_back(const_cast<char *>(file.c_str()));
    }
    c_files.push_back(nullptr); // 以nullptr结尾

    try
    {
        if ((kind == "base" || kind == "runtime") && !d->m_isHostapp)
        {
            kmlogger.info("setLabelForKaimingInterpreter for base or runtime");
            auto setLabelForKaimingInterpreter = ksafLabelHandle.loadSymbol<SetLabelForInterpreterFunc>("setLabelForKaimingInterpreter");

            if (int ret = setLabelForKaimingInterpreter(path.c_str()); ret != 0)
            {
                kmlogger.error("Failed to 'setLabelForKaimingInterpreter': %d, path: %s", ret, path.c_str());
                return KM_ERROR_SETLABEL_FAILED;
            }
        }
        else
        {
            auto setLabelForKaiming = ksafLabelHandle.loadSymbol<SetLabelFunc>("setLabelForKaiming");

            if (int ret = setLabelForKaiming(c_files.data(), appName.c_str()); ret != 0)
            {
                kmlogger.error("Failed to 'setLabelForKaiming' (code: %d)", ret);
                return KM_ERROR_SETLABEL_FAILED;
            }
        }

        kmlogger.info("set label successfully");

        auto exectl_xattr_multi_set = exectlHandle.loadSymbol<XattrSetFunc>("exectl_xattr_multi_set");

        int ret = exectl_xattr_multi_set(c_paths.data(), c_paths.size());
        if (ret < 0)
        {
            kmlogger.error("Failed to set xattr (code: %d)", ret);
            return KM_ERROR_SETLABEL_FAILED;
        }

        kmlogger.info("set exectl xattr successfully (code: %d)", ret);
    }
    catch (...)
    {
        kmlogger.error("setKsafLabelAndExectl crashed unexpectedly");
        return KM_ERROR_SETLABEL_FAILED;
    }

    kmlogger.info("setKsafLabelAndExectl completed successfully");
    return KM_ERROR_NO;
}

int KMPackageManager::getSafeMode()
{
    void *handle = dlopen("libkylin_signtool.so", RTLD_LAZY);
    if (!handle)
    {
        kmlogger.error("Failed to load libkylin_signtool.so: %s", dlerror());
        return 0;
    }

    typedef int (*GetDpkgInfoFunc)();
    GetDpkgInfoFunc get_dpkg_info = reinterpret_cast<GetDpkgInfoFunc>(
        dlsym(handle, "get_dpkg_file_info"));

    if (!get_dpkg_info)
    {
        kmlogger.error("Failed to find get_dpkg_file_info: %s", dlerror());
        dlclose(handle);
        return 0;
    }

    int result = get_dpkg_info();

    dlclose(handle);

    return result;
}

int KMPackageManager::resolveFailedVerification(int safeStatus, fs::path idPath, std::string id)
{
    switch (safeStatus)
    {
    case 1:
        kmlogger.info("安全模式为关闭模式，跳过验签");
        return KM_ERROR_NO;

    case 2:
    {
        void *handle = dlopen("libkysec.so", RTLD_LAZY);
        if (!handle)
        {
            kmlogger.error("Failed to load libkysec.so: %s", dlerror());
            return KM_ERROR_SIGN_FAILED;
        }

        dlerror();

        using SignCheckFunc = int (*)(const char *, const char *);
        SignCheckFunc kysec_check_custom_pkg_sign = (SignCheckFunc)dlsym(handle, "kysec_check_custom_pkg_sign");

        const char *dlsym_error = dlerror();
        if (dlsym_error)
        {
            kmlogger.error("Failed to find kysec_check_pkg_sign: %s", dlsym_error);
            dlclose(handle);
            return KM_ERROR_SIGN_FAILED;
        }

        // 返回值 1:允许安装;0:禁止安装
        int signStatus = kysec_check_custom_pkg_sign(const_cast<char *>(id.c_str()), const_cast<char *>(idPath.c_str()));
        kmlogger.info("User selection after popup: %s", signStatus ? "允许安装" : "禁止安装");

        dlclose(handle);
        return signStatus == 1 ? KM_ERROR_NO : KM_ERROR_USER_ABORTED;
    }

    case 3:
        kmlogger.error("安全模式为阻止模式，验签失败，拒绝安装");
        return KM_ERROR_SIGN_FAILED;

    default:
        kmlogger.error("获取安全模式失败");
        return KM_ERROR_GET_SAFE_MODE_FAILED;
    }
}

int KMPackageManager::pullFromOstree(fs::path path, KMPMRef downloadRef)
{
    fs::path tmpPath = path;
    kmlogger.info("Begin to deploy temporary files to /tmp.");

    // d->m_installedFiles.clear();
    kmlogger.info("completedRef: %s", downloadRef.completedRef.c_str());

    fs::path idPath = tmpPath / downloadRef.ref.id;
    kmlogger.debug("idPath:  %s", idPath.c_str());
    // d->m_installedFiles.insert(idPath.string());

    if (std::filesystem::exists(idPath) || std::filesystem::create_directories(idPath))
    {
        kmlogger.debug("Directory created or already exists successfully!");
    }

    kmlogger.debug("Begin to checkout ...");
    string latestCommit = d->m_ostreeHandler->readLatestCommit(downloadRef.completedRef);
    // 生成files.md5等文件,从 OSTree 仓库中检出（checkout）指定的提交（commit），并将内容部署到指定的路径
    d->m_ostreeHandler->checkout(latestCommit, idPath.string());
    kmlogger.debug("End to checkout.");
    kmlogger.info("End to deploy files.");

    return 0;
}

string KMPackageManager::getRemoteUrl(const string &remoteName)
{
    return d->m_ostreeHandler->getRemoteUrl(remoteName);
}

bool KMPackageManager::clearCacheFiles(const string &id)
{
    const fs::path infoPath = d->m_folder->getInfoPath();
    const fs::path infoListFile = infoPath / (id + ".list");
    const fs::path binListFile = infoPath / (id + ".bin");

    std::error_code ec;
    bool allRemoved = true;

    auto removeFile = [&allRemoved](const fs::path& file) {
        std::error_code ec;
        if (!fs::remove(file, ec)) {
            if (ec.value() != static_cast<int>(std::errc::no_such_file_or_directory)) {
                kmlogger.debug("Failed to remove %s: %s", file.string().c_str(), ec.message().c_str());
                allRemoved = false;
            }
        }
        return !ec;
    };

    removeFile(infoListFile);
    removeFile(binListFile);

    return allRemoved;
}

bool KMPackageManager::removeHostAppDeployFiles(const std::string &strPath)
{
    std::string packageId = KMStringUtils::extractPackageId(strPath);
    if (!packageId.empty())
    {
        d->m_hostAppDeploy->setHostAppConfig(strPath, packageId);
        if (d->m_hostAppDeploy->loadHostAppInfo())
        {
            d->m_hostAppDeploy->uninstall();
        }

        clearCacheFiles(packageId);
    }

    return true;
}

void KMPackageManager::removeHardlinksInPlace(const fs::path& dir) {
    try {
        std::unordered_set<ino_t> processed_inodes;

        for (const auto& entry : fs::recursive_directory_iterator(dir))
        {
            if (fs::is_regular_file(entry))
            {
                // 正确获取 inode 的方式
                struct stat file_stat;
                if (stat(entry.path().c_str(), &file_stat) == 0)
                {
                    ino_t inode = file_stat.st_ino;
                    
                    if (processed_inodes.find(inode) == processed_inodes.end()) 
                    {
                        processed_inodes.insert(inode);
                    }
                    else
                    {
                        // 处理硬链接：创建临时副本
                        fs::path temp_file = entry.path().string() + ".tmp";
                        
                        // 复制内容
                        std::ifstream src(entry.path(), std::ios::binary);
                        std::ofstream dst(temp_file, std::ios::binary);
                        dst << src.rdbuf();
                        src.close();
                        dst.close();
                        
                        // 替换原文件
                        fs::rename(temp_file, entry.path());
                    }
                }
            }
        }
    }
    catch (const fs::filesystem_error& e)
    {
        std::cerr << "Error: " << e.what() << std::endl;
    }
}