/*
 * Copyright (c) KylinSoft  Co., Ltd. 2024. All rights reserved.
 *
 * kaiming is licensed under the GPL v2.0+.
 * 
 * See the LICENSE file for more details.
 */

#ifndef __KMINFOJSON_H__
#define __KMINFOJSON_H__

#include "KMJsonHelper.h"
#include "KMCommonTypePrivate.h"

struct Permissions
{
    bool all = false;
    bool xdgUserDirs = false;
    bool network = false;
    bool scanning = false;
    bool bluetooth = false;
    bool microphone = false;
    bool camera = false;
    bool usb = false;
    bool serial = false;
    bool autostart = false;
    bool systemNotice = false;
    bool systemTime = false;
    bool systemServer = false;
    bool windowTop = false;
    bool preventSleep = false;
    bool systemDisplayResolution = false;
    bool systemDisplayBrightness = false;
    bool systemAudioSettings = false;
    bool systemAudioSize = false;
    bool systemCrontab = false;
    bool systemPowerMode = false;
    bool systemPowerPolicy = false;
    std::vector<std::string> filesystemDirs;
};

struct Annotations
{
    bool privilege = false;
    bool sysapp = false;
    std::vector<std::string> depends;
    bool hostapp = false;
    std::vector<std::string> service;
    std::vector<std::string> sysgroup;
    std::vector<std::string> dependApps;
};

/**
 * @brief : 读写info.json配置文件的类
 */
class KMInfoJson
{
public:
    /**
     * @brief : 解析info中的base部分
     */
    class InfoBase
    {
    public:
        InfoBase() = default;
        ~InfoBase() = default;
        /**
         * @brief : 解析base获取id与version
         */
        void baseParse();
        /**
         * @brief : 解析version
         */
        void versionParse();
        /**
         * @brief : 检查版本号是否匹配
         * @param : [in] ver 待比较的版本号
         * @return : true-匹配；false-不匹配
         */
        bool checkVersion(const InfoBase &ver);

    public:
        std::string m_baseString;
        std::string m_baseId;
        std::string m_baseVersion;
        std::string m_versionMajor;
        std::string m_versionMinor;
    };

public:
    KMInfoJson() = default;
    ~KMInfoJson() = default;

    /**
     * @brief : 加载json文件
     * @param : [in] jsonFile, json文件
     * @return: nlohmann::json, 解析后的json对象
     * @throw : KMException
     */
    void loadFile(const std::string &jsonFile);

    /**
     * @brief : 保存json对象到文件
     * @param : [in] jsonFile, json文件
     * @throw : KMException
     */
    void saveFile(const std::string &jsonFile);

    void setRoot(nlohmann::json &root);

    nlohmann::json root();

    /**
     * @brief : 解析JSON中的permission字段
     * @param : 包含权限信息的nlohmann::json对象
     * @return : void
     */
    void parsePermissions();

    /**
     * @brief 获取已解析的权限信息
     * @return const Permissions& 返回对已解析权限结构体的常量引用
     */
    const Permissions &getPermissions() const;

    /**
     * @brief : 解析JSON中的annotations字段
     * @param : 包含自定义信息
     * @return : void
     */
    void parseAnnotations();

    /**
     * @brief 获取已解析的权限信息
     * @return const Permissions& 返回对已解析权限结构体的常量引用
     */
    const Annotations getAnnotations() const;

    KMRef baseRef() const;

    KMRef runtimeRef() const;

    static KMRef parseKMRef(const std::string &ref);

    std::string& operator[](const std::string& key);

private:
    void fromJson();

    void toJson();

    void permissionToJson();

    void annotationsToJson();

public:
    // info.json属性域，暂时不进行封装

    // 协议版本号
    std::string schema_version = "1.0";

    // 应用id
    std::string id;

    // 架构
    std::vector<std::string> archs;

    // base
    std::string base;

    // 运行时
    std::string runtime;

    std::string depend;
    
    // 渠道
    std::string channel;

    // kaiming run执行命令
    std::vector<std::string> commands;

    // 应用名称
    std::string name;

    // 多语言应用名称，例：name[zh_CN]
    std::map<std::string, std::string> names;

    // 描述
    std::string description;

    // 多语言描述，例：description[zh_CN]
    std::map<std::string, std::string> descriptions;

    // kind {app/runtime/base}
    std::string kind;

    // module (binary、devel)
    std::string module;

    // 安装大小
    uint64_t size = 0;

    // 版本
    std::string version;

    // 环境变量
    std::map<std::string, std::string> envs;

    // 权限
    Permissions permissions;

    // 许可证信息，SPDX格式
    std::string license;

    // 首页
    std::string homepage;

    // 维护者
    std::vector<std::string> maintainers;

    // 自定义
    Annotations annotations;

private:
    // 数据成员域
    nlohmann::json m_root;
};

#endif // !__KMINFOJSON_H__
