#ifndef KMCOMMON_UTILS_H
#define KMCOMMON_UTILS_H

#include <gio/gio.h>
#include <sys/wait.h>
#include <string>

#include "common/KMLogger.h"

static void runGtaskCommand(const std::string &command, GTask *task)
{
    int pipeFD[2];
    if (pipe(pipeFD) == -1)
    {
        g_task_return_new_error(task, G_IO_ERROR, G_IO_ERROR_FAILED, "create pipe failed");
        return;
    }

    auto pipeGuard = std::unique_ptr<int, void (*)(int *)>(
        pipeFD, [](int *fds) {
            close(fds[0]);
            close(fds[1]);
        });

    pid_t pid = fork();
    if (pid == -1)
    {
        g_task_return_new_error(task, G_IO_ERROR, G_IO_ERROR_FAILED, "fork failed");
        return;
    }

    if (pid == 0)
    {
        close(pipeFD[0]);
        dup2(pipeFD[1], STDOUT_FILENO);
        dup2(pipeFD[1], STDERR_FILENO);
        close(pipeFD[1]);
        const char *sh = "/bin/bash";
        const char *args[] = { sh, "-c", command.c_str(), nullptr };
        execv(sh, const_cast<char *const *>(args));
        KMError("exec command failed !");
        exit(EXIT_FAILURE);
    }
    else
    {
        close(pipeFD[1]);
        GString *output = g_string_new(nullptr);
        char buffer[256];
        ssize_t count;

        while ((count = read(pipeFD[0], buffer, sizeof(buffer) - 1)) > 0)
        {
            buffer[count] = '\0';
            g_string_append(output, buffer);
        }
        close(pipeFD[0]);

        int status = 0;
        waitpid(pid, &status, 0);
        if (WIFEXITED(status) && WEXITSTATUS(status) != 0)
        {
            g_string_append_printf(output, "\nexit code: %d\n", WEXITSTATUS(status));
        }

        g_task_return_pointer(task, g_string_free(output, FALSE), g_free);
    }
}

#endif // KMCOMMON_UTILS_H