/*
 * Copyright (c) KylinSoft  Co., Ltd. 2024. All rights reserved.
 *
 * kaiming is licensed under the GPL v2.0+.
 * 
 * See the LICENSE file for more details.
 */

#include "KMUpdate.h"

#include <fcntl.h>
#include <sys/personality.h>
#include <grp.h>
#include <pwd.h>

#include "app/list/KMList.h"
#include "common/KMPackageManager.h"
#include "common/KMError.h"
#include "common/KMLogger.h"
#include "common/KMUtil.h"
#include "common/KMStringUtils.h"

/**
 * @brief 更新子命令的参数解析
 */
class KMUpdate::Options : public KMOption::Options
{
public:
    Options() = default;
    ~Options() override = default;

    void checkUnknownOptions(int argc, char** argv);

protected:
    /**
     * @brief 命令行参数解析后操作
     */
    void preParseHook() override;

    /**
     * @brief 命令行参数解析后操作
     */
    void postParseHook() override;

private:
    /**
     * @brief 命令行参数添加描述选项
     */
    void addOptions();

public:
    bool m_help;
    bool m_assumeYes;
    string m_arch;
    bool m_upgradable;
    vector<string> m_vecRefs;
};

void KMUpdate::Options::preParseHook()
{
    addOptions();
}

/**
 * @brief 命令行参数解析后操作
 */
void KMUpdate::Options::postParseHook()
{
    if (m_help)
    {
        showUsage();
        exit(EXIT_SUCCESS);
    }
}

/**
 * @brief 命令行参数添加描述选项
 */
void KMUpdate::Options::addOptions()
{
    setDescription(_("\nUsage:\n \tkaiming update [options] [quote…] - upgrade package description\n"));
    //帮助选项
    addOption("help", "h", KMOption::value<bool>(&m_help)->defaultValue(false), _("display Help Options"));
    addOption("assumeyes", "y", KMOption::value<bool>(&m_assumeYes)->defaultValue(false), _("automatic answer to all questions is yes"));
    addOption("upgradable", "", KMOption::value<bool>(&m_upgradable)->defaultValue(false), _("List all upgradable applications"));

    //应用选项
    addOption("arch", "", KMOption::value<std::string>(&m_arch), _("specify architecture"));
    //位置选项
    addPositionOption("REFS", KMOption::value<std::vector<std::string>>(&m_vecRefs), -1, _("specify the application REF"));
}

class KMUpdate::Private
{
public:
    std::shared_ptr<Options> m_options;
    std::shared_ptr<KMPackageManager> m_packageManager;
    std::string m_arch;
};

REGISTER_SUBCOMMAND_DYNCREATE(update, KMUpdate)

KMUpdate::KMUpdate()
    : d(std::make_unique<Private>())
{
    init();
}

KMUpdate::~KMUpdate() = default;

int KMUpdate::dispose(int argc, char **argv)
{
    d->m_options->checkUnknownOptions(argc, argv); 
    vector<string> vecUpdate;
    for (int i = 1; i < argc; i++)
    {
        if (argv[i][0] != '-')
        {
            vecUpdate.emplace_back(argv[i]);
        }
        else if (!vecUpdate.empty())
        {
            kmlogger.error("args error");
            return KM_ERROR_INVALID_ARGS;
        }
    }

    // 参数解析
    d->m_options->parseCommandLine(argc, argv);

    int ret = start(vecUpdate);

    return ret;
}

/**
 * @brief 开始对已获取的用户软件包列表执行安装
 *
 * @param vecName
 * @return int
 */
int KMUpdate::start(const vector<string> &vecName)
{
    int ret = 0;
    if (vecName.empty())
    {
        std::cout << _("Search matching packages... ") << std::endl;
        KMList listIds;
        std::map<std::string, std::string> localAppsInfo = listIds.getLocalAppIds(false);

        PackageManagerOptions pmOptions;
        pmOptions.assumeYes = d->m_options->m_assumeYes;
        pmOptions.arch = d->m_options->m_arch;
        pmOptions.upgradable = d->m_options->m_upgradable;
        pmOptions.isGetDevel = true;
        pmOptions.isPrint = false;
        ret = d->m_packageManager->getUpgradableList(localAppsInfo, pmOptions);
    }
    else
    {
        PackageManagerOptions pmOptions;
        pmOptions.assumeYes = d->m_options->m_assumeYes;
        pmOptions.arch = d->m_options->m_arch;

        ret = d->m_packageManager->processTask(TASK_TYPE_UPDATE, vecName, pmOptions);
    }

    return ret;
}

void KMUpdate::init()
{
    d->m_options = std::make_shared<Options>();
    d->m_packageManager = std::make_shared<KMPackageManager>();
}

void KMUpdate::Options::checkUnknownOptions(int argc, char** argv)
{
    std::set<std::string> validOptions = {
        "--help", "-h",
        "--assumeyes", "-y",
        "--upgradable",
        "--arch"
    };

    for (int i = 1; i < argc; ++i) 
    {
        std::string arg(argv[i]);

        if (arg.size() >= 1 && arg[0] == '-') 
        {
            std::string opt = arg;
            auto eq_pos = opt.find('=');
            if (eq_pos != std::string::npos)
            {
                opt = opt.substr(0, eq_pos);
            }

            if (validOptions.find(opt) == validOptions.end()) 
            {
                KMError(_("Unrecognized option “") + arg + "”");
                std::cerr << _("Please use ") << ("'kaiming update --help'") << _(" to see available options.") << std::endl;
                exit(EXIT_FAILURE);
            }
        }
    }
}