/*
 * Copyright (c) KylinSoft  Co., Ltd. 2024. All rights reserved.
 *
 * kaiming is licensed under the GPL v2.0+.
 * 
 * See the LICENSE file for more details.
 */

#include "KMPS.h"

#include <map>

#include "common/KMLogger.h"
#include "common/KMUtil.h"
#include "common/KMConfigData.h"
#include "common/KMCrun.h"

using namespace std;

/**
 * @brief 安装子命令的参数解析
 */
class KMPS::Options : public KMOption::Options
{
public:
    Options() = default;
    ~Options() override= default;

    void checkUnknownOptions(int argc, char** argv);

protected:
    /**
     * @brief 命令行参数解析后操作
     */
    void preParseHook() override;

    /**
     * @brief 命令行参数解析后操作
     */
    void postParseHook() override;

private:
    /**
     * @brief 命令行参数添加描述选项
     */
    void addOptions();

public:
    bool m_bHelp;
    vector<string> m_vecRefs;
};

void KMPS::Options::preParseHook()
{
    addOptions();
}

/**
 * @brief 命令行参数解析后操作
 */
void KMPS::Options::postParseHook()
{
    if (m_bHelp)
    {
        showUsage();
        exit(EXIT_SUCCESS);
    }
}
/**
 * @brief 命令行参数添加描述选项
 */
void KMPS::Options::addOptions()
{
    setDescription(_("\nUsage:\n \tkaiming ps [options]  - list the running container applications\n"));
    addOption("help","h",KMOption::value<bool>(&m_bHelp),_("display help options"));
}

class KMPS::Private
{
public:
    std::shared_ptr<Options> m_options;
};

REGISTER_SUBCOMMAND_DYNCREATE(ps, KMPS)

KMPS::KMPS()
    : m_dPtr(make_shared<Private>())
{
    init();
}

KMPS::~KMPS() = default;

/**
 * @brief ps 命令参数处理,不需要参数
 * 列出所有正在运行的容器应用,只支持不含参数
 * 使用实例:
 * kaiming ps
 * @param argc 参数个数
 * @param argv 参数列表
 * @return int 0:成功,非0:失败
 */

int KMPS::dispose(int argc, char **argv)
{
    m_dPtr->m_options->checkUnknownOptions(argc, argv);
    m_dPtr->m_options->parseCommandLine(argc, argv);

    vector<ContainerListItem> item;
    vector<string> args = { "list", "--format=json" };
    if (!KMCrun::exec<vector<ContainerListItem>>(args, item))
    {
        return EXIT_FAILURE;
    }

    std::vector<KMPSInfoList> psInfos;
    for (const auto &i : item)
    {
        // 获取信息
        KMPSInfoList psInfo;
        psInfo.containerid = i.id;
        psInfo.pid = i.pid;
        psInfo.path = i.bundle;
        State itemAppid;
        vector<string> argsAppid = { "state", psInfo.containerid };
        if (!KMCrun::exec<State>(argsAppid, itemAppid))
        {
            continue;
        }

        psInfo.appid = itemAppid.annotations.value_or(std::map<std::string, std::string>{})["kaiming.app.id"];
        psInfos.emplace_back(psInfo);
    }

    printAppInfo(psInfos);

    return EXIT_SUCCESS;
}

void KMPS::init()
{
    m_dPtr->m_options = make_shared<Options>();
}

/**
 * @brief 格式化输出包信息
 */
void KMPS::printAppInfo(const vector<KMPSInfoList> &psInfos) const
{
    // 打印表头
    printf("%-30s %-40s %-10s %-80s\n",_("appid"),_("containerid"), _("pid"),_("path"));

    // 遍历数组并打印每个元素
    for (const auto &psInfo : psInfos)
    {
        printf(_("%-30s %-40s %-10s %-80s\n"),
               psInfo.appid.c_str(),
               psInfo.containerid.c_str(),
               to_string(psInfo.pid).c_str(),
               psInfo.path.c_str());
    }
}

void KMPS::Options::checkUnknownOptions(int argc, char** argv)
{
    std::set<std::string> validOptions = {
        "--help", "-h"
    };

    for (int i = 1; i < argc; ++i) 
    {
        std::string arg(argv[i]);

        if (arg.size() >= 1 && arg[0] == '-') 
        {
            std::string opt = arg;
            auto eq_pos = opt.find('=');
            if (eq_pos != std::string::npos)
            {
                opt = opt.substr(0, eq_pos);
            }

            if (validOptions.find(opt) == validOptions.end()) 
            {
                KMError(_("Unrecognized option “") + arg + "”");
                std::cerr << _("Please use ") << ("'kaiming ps --help'") << _(" to see available options.") << std::endl;
                exit(EXIT_FAILURE);
            }
        }
    }
}