//
// Copyright 2016 Pixar
//
// Licensed under the terms set forth in the LICENSE.txt file available at
// https://openusd.org/license.
//
#ifndef USDGEOM_GENERATED_NURBSCURVES_H
#define USDGEOM_GENERATED_NURBSCURVES_H

/// \file usdGeom/nurbsCurves.h

#include "pxr/pxr.h"
#include "pxr/usd/usdGeom/api.h"
#include "pxr/usd/usdGeom/curves.h"
#include "pxr/usd/usd/prim.h"
#include "pxr/usd/usd/stage.h"
#include "pxr/usd/usdGeom/tokens.h"

#include "pxr/base/vt/value.h"

#include "pxr/base/gf/vec3d.h"
#include "pxr/base/gf/vec3f.h"
#include "pxr/base/gf/matrix4d.h"

#include "pxr/base/tf/token.h"
#include "pxr/base/tf/type.h"

PXR_NAMESPACE_OPEN_SCOPE

class SdfAssetPath;

// -------------------------------------------------------------------------- //
// NURBSCURVES                                                                //
// -------------------------------------------------------------------------- //

/// \class UsdGeomNurbsCurves
///
/// This schema is analagous to NURBS Curves in packages like Maya
/// and Houdini, often used for interchange of rigging and modeling curves.  
/// Unlike Maya, this curve spec supports batching of multiple curves into a 
/// single prim, widths, and normals in the schema.  Additionally, we require 
/// 'numSegments + 2 * degree + 1' knots (2 more than maya does).  This is to
/// be more consistent with RenderMan's NURBS patch specification.  
/// 
/// To express a periodic curve:
/// - knot[0] = knot[1] - (knots[-2] - knots[-3]; 
/// - knot[-1] = knot[-2] + (knot[2] - knots[1]);
/// 
/// To express a nonperiodic curve:
/// - knot[0] = knot[1];
/// - knot[-1] = knot[-2];
/// 
/// In spite of these slight differences in the spec, curves generated in Maya
/// should be preserved when roundtripping.
/// 
/// \em order and \em range, when representing a batched NurbsCurve should be
/// authored one value per curve.  \em knots should be the concatentation of
/// all batched curves.
///
class UsdGeomNurbsCurves : public UsdGeomCurves
{
public:
    /// Compile time constant representing what kind of schema this class is.
    ///
    /// \sa UsdSchemaKind
    static const UsdSchemaKind schemaKind = UsdSchemaKind::ConcreteTyped;

    /// Construct a UsdGeomNurbsCurves on UsdPrim \p prim .
    /// Equivalent to UsdGeomNurbsCurves::Get(prim.GetStage(), prim.GetPath())
    /// for a \em valid \p prim, but will not immediately throw an error for
    /// an invalid \p prim
    explicit UsdGeomNurbsCurves(const UsdPrim& prim=UsdPrim())
        : UsdGeomCurves(prim)
    {
    }

    /// Construct a UsdGeomNurbsCurves on the prim held by \p schemaObj .
    /// Should be preferred over UsdGeomNurbsCurves(schemaObj.GetPrim()),
    /// as it preserves SchemaBase state.
    explicit UsdGeomNurbsCurves(const UsdSchemaBase& schemaObj)
        : UsdGeomCurves(schemaObj)
    {
    }

    /// Destructor.
    USDGEOM_API
    virtual ~UsdGeomNurbsCurves();

    /// Return a vector of names of all pre-declared attributes for this schema
    /// class and all its ancestor classes.  Does not include attributes that
    /// may be authored by custom/extended methods of the schemas involved.
    USDGEOM_API
    static const TfTokenVector &
    GetSchemaAttributeNames(bool includeInherited=true);

    /// Return a UsdGeomNurbsCurves holding the prim adhering to this
    /// schema at \p path on \p stage.  If no prim exists at \p path on
    /// \p stage, or if the prim at that path does not adhere to this schema,
    /// return an invalid schema object.  This is shorthand for the following:
    ///
    /// \code
    /// UsdGeomNurbsCurves(stage->GetPrimAtPath(path));
    /// \endcode
    ///
    USDGEOM_API
    static UsdGeomNurbsCurves
    Get(const UsdStagePtr &stage, const SdfPath &path);

    /// Attempt to ensure a \a UsdPrim adhering to this schema at \p path
    /// is defined (according to UsdPrim::IsDefined()) on this stage.
    ///
    /// If a prim adhering to this schema at \p path is already defined on this
    /// stage, return that prim.  Otherwise author an \a SdfPrimSpec with
    /// \a specifier == \a SdfSpecifierDef and this schema's prim type name for
    /// the prim at \p path at the current EditTarget.  Author \a SdfPrimSpec s
    /// with \p specifier == \a SdfSpecifierDef and empty typeName at the
    /// current EditTarget for any nonexistent, or existing but not \a Defined
    /// ancestors.
    ///
    /// The given \a path must be an absolute prim path that does not contain
    /// any variant selections.
    ///
    /// If it is impossible to author any of the necessary PrimSpecs, (for
    /// example, in case \a path cannot map to the current UsdEditTarget's
    /// namespace) issue an error and return an invalid \a UsdPrim.
    ///
    /// Note that this method may return a defined prim whose typeName does not
    /// specify this schema class, in case a stronger typeName opinion overrides
    /// the opinion at the current EditTarget.
    ///
    USDGEOM_API
    static UsdGeomNurbsCurves
    Define(const UsdStagePtr &stage, const SdfPath &path);

protected:
    /// Returns the kind of schema this class belongs to.
    ///
    /// \sa UsdSchemaKind
    USDGEOM_API
    UsdSchemaKind _GetSchemaKind() const override;

private:
    // needs to invoke _GetStaticTfType.
    friend class UsdSchemaRegistry;
    USDGEOM_API
    static const TfType &_GetStaticTfType();

    static bool _IsTypedSchema();

    // override SchemaBase virtuals.
    USDGEOM_API
    const TfType &_GetTfType() const override;

public:
    // --------------------------------------------------------------------- //
    // ORDER 
    // --------------------------------------------------------------------- //
    /// Order of the curve.  Order must be positive and is
    /// equal to the degree of the polynomial basis to be evaluated, plus 1.
    /// Its value for the 'i'th curve must be less than or equal to
    /// curveVertexCount[i]
    ///
    /// | ||
    /// | -- | -- |
    /// | Declaration | `int[] order = []` |
    /// | C++ Type | VtArray<int> |
    /// | \ref Usd_Datatypes "Usd Type" | SdfValueTypeNames->IntArray |
    USDGEOM_API
    UsdAttribute GetOrderAttr() const;

    /// See GetOrderAttr(), and also 
    /// \ref Usd_Create_Or_Get_Property for when to use Get vs Create.
    /// If specified, author \p defaultValue as the attribute's default,
    /// sparsely (when it makes sense to do so) if \p writeSparsely is \c true -
    /// the default for \p writeSparsely is \c false.
    USDGEOM_API
    UsdAttribute CreateOrderAttr(VtValue const &defaultValue = VtValue(), bool writeSparsely=false) const;

public:
    // --------------------------------------------------------------------- //
    // KNOTS 
    // --------------------------------------------------------------------- //
    /// Knot vector providing curve parameterization.
    /// The length of the slice of the array for the ith curve 
    /// must be ( curveVertexCount[i] + order[i] ), and its
    /// entries must take on monotonically increasing values.
    ///
    /// | ||
    /// | -- | -- |
    /// | Declaration | `double[] knots` |
    /// | C++ Type | VtArray<double> |
    /// | \ref Usd_Datatypes "Usd Type" | SdfValueTypeNames->DoubleArray |
    USDGEOM_API
    UsdAttribute GetKnotsAttr() const;

    /// See GetKnotsAttr(), and also 
    /// \ref Usd_Create_Or_Get_Property for when to use Get vs Create.
    /// If specified, author \p defaultValue as the attribute's default,
    /// sparsely (when it makes sense to do so) if \p writeSparsely is \c true -
    /// the default for \p writeSparsely is \c false.
    USDGEOM_API
    UsdAttribute CreateKnotsAttr(VtValue const &defaultValue = VtValue(), bool writeSparsely=false) const;

public:
    // --------------------------------------------------------------------- //
    // RANGES 
    // --------------------------------------------------------------------- //
    /// Provides the minimum and maximum parametric values (as defined
    /// by knots) over which the curve is actually defined.  The minimum must 
    /// be less than the maximum, and greater than or equal to the value of the 
    /// knots['i'th curve slice][order[i]-1]. The maxium must be less 
    /// than or equal to the last element's value in knots['i'th curve slice].
    /// Range maps to (vmin, vmax) in the RenderMan spec.
    ///
    /// | ||
    /// | -- | -- |
    /// | Declaration | `double2[] ranges` |
    /// | C++ Type | VtArray<GfVec2d> |
    /// | \ref Usd_Datatypes "Usd Type" | SdfValueTypeNames->Double2Array |
    USDGEOM_API
    UsdAttribute GetRangesAttr() const;

    /// See GetRangesAttr(), and also 
    /// \ref Usd_Create_Or_Get_Property for when to use Get vs Create.
    /// If specified, author \p defaultValue as the attribute's default,
    /// sparsely (when it makes sense to do so) if \p writeSparsely is \c true -
    /// the default for \p writeSparsely is \c false.
    USDGEOM_API
    UsdAttribute CreateRangesAttr(VtValue const &defaultValue = VtValue(), bool writeSparsely=false) const;

public:
    // --------------------------------------------------------------------- //
    // POINTWEIGHTS 
    // --------------------------------------------------------------------- //
    /// Optionally provides "w" components for each control point,
    /// thus must be the same length as the points attribute.  If authored,
    /// the curve will be rational.  If unauthored, the curve will be
    /// polynomial, i.e. weight for all points is 1.0.
    /// \note Some DCC's pre-weight the \em points, but in this schema, 
    /// \em points are not pre-weighted.
    ///
    /// | ||
    /// | -- | -- |
    /// | Declaration | `double[] pointWeights` |
    /// | C++ Type | VtArray<double> |
    /// | \ref Usd_Datatypes "Usd Type" | SdfValueTypeNames->DoubleArray |
    USDGEOM_API
    UsdAttribute GetPointWeightsAttr() const;

    /// See GetPointWeightsAttr(), and also 
    /// \ref Usd_Create_Or_Get_Property for when to use Get vs Create.
    /// If specified, author \p defaultValue as the attribute's default,
    /// sparsely (when it makes sense to do so) if \p writeSparsely is \c true -
    /// the default for \p writeSparsely is \c false.
    USDGEOM_API
    UsdAttribute CreatePointWeightsAttr(VtValue const &defaultValue = VtValue(), bool writeSparsely=false) const;

public:
    // ===================================================================== //
    // Feel free to add custom code below this line, it will be preserved by 
    // the code generator. 
    //
    // Just remember to: 
    //  - Close the class declaration with }; 
    //  - Close the namespace with PXR_NAMESPACE_CLOSE_SCOPE
    //  - Close the include guard with #endif
    // ===================================================================== //
    // --(BEGIN CUSTOM CODE)--
};

PXR_NAMESPACE_CLOSE_SCOPE

#endif
