//! Sysfs attribute reading utilities.

use std::ffi::OsString;
use std::fs::File;
use std::io::{self, ErrorKind, Read};
use std::os::unix::ffi::OsStringExt;
use std::path::Path;

/// Maximum size for sysfs attribute reads.
/// This should be larger than any expected identifier.
const MAX_ATTR_SIZE: usize = 256;

/// Read a sysfs attribute file, returning its contents exactly as read.
///
/// The bytes are returned exactly as read from sysfs, with no trimming
/// or conversion applied.
///
/// # Errors
///
/// Returns `Err` if:
/// - The file cannot be opened or read
/// - The attribute is larger than our buffer (would be truncated)
pub fn read_attr<P: AsRef<Path>>(path: P) -> Result<OsString, ReadAttrError> {
    let mut file = File::open(path.as_ref())?;

    let mut buf = [0u8; MAX_ATTR_SIZE];
    let mut total = 0;

    // Read until EOF or buffer full, handling EINTR
    loop {
        if total >= buf.len() {
            break;
        }

        match file.read(&mut buf[total..]) {
            Ok(0) => break, // EOF
            Ok(n) => total += n,
            Err(e) if e.kind() == ErrorKind::Interrupted => continue,
            Err(e) => return Err(ReadAttrError::Io(e)),
        }
    }

    // Check if we hit EOF or if there's more data (truncation)
    if total >= buf.len() {
        // Buffer is full - check if there's more data
        let mut probe = [0u8; 1];
        loop {
            match file.read(&mut probe) {
                Ok(0) => break,                // Exactly at EOF, no truncation
                Ok(_) => return Err(ReadAttrError::Truncated), // More data exists
                Err(e) if e.kind() == ErrorKind::Interrupted => continue,
                Err(e) => return Err(ReadAttrError::Io(e)),
            }
        }
    }

    Ok(OsString::from_vec(buf[..total].to_vec()))
}

/// Errors from reading sysfs attributes.
#[derive(Debug)]
pub enum ReadAttrError {
    /// An I/O error occurred.
    Io(io::Error),
    /// The attribute was too large and would have been truncated.
    Truncated,
}

impl From<io::Error> for ReadAttrError {
    fn from(e: io::Error) -> Self {
        ReadAttrError::Io(e)
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use std::io::Write;
    use std::os::unix::ffi::OsStrExt;
    use tempfile::NamedTempFile;

    #[test]
    fn test_read_attr_simple() {
        let mut file = NamedTempFile::new().unwrap();
        writeln!(file, "test_value").unwrap();

        let result = read_attr(file.path()).unwrap();
        assert_eq!(result.as_bytes(), b"test_value\n");
    }

    #[test]
    fn test_read_attr_preserves_whitespace() {
        let mut file = NamedTempFile::new().unwrap();
        write!(file, "  value  \n\n").unwrap();

        let result = read_attr(file.path()).unwrap();
        assert_eq!(result.as_bytes(), b"  value  \n\n");
    }
}
