# block-device-id

A Rust library and command-line tool to retrieve hardware-innate unique identifiers from block devices, such as MMC CID or NVME controller serial numbers. These identifiers are suitable for use in cryptographic key derivation or device-specific configuration.

## Features

- C-compatible shared library interface
- Command-line tool for querying device identifiers
- Supports multiple block device types (MMC, NVME, etc.)
- Debian-native package with vendored dependencies

## Building

### Standard Build

To build the Debian package normally:

```bash
debuild -b -uc -us
```

### Network-Isolated Build with bwrap

This is a Debian-native Rust package that uses vendored Debian versions of packages rather than fetching from crates.io. For network-isolated builds (e.g., in build environments without internet access), you can use `bwrap` (bubblewrap) to create an isolated build environment.

#### Prerequisites

- `bwrap` must be installed
- A `resolv.conf` file must be present in the project root (or create one for DNS resolution if needed)

#### Build Command

```bash
bwrap --bind / / --bind ./resolv.conf /etc/resolv.conf --dev /dev --proc /proc -- debuild -b -uc -us
```

This command:
- `--bind / /` - Binds the root filesystem
- `--bind ./resolv.conf /etc/resolv.conf` - Uses a local resolv.conf file (prevents DNS lookups)
- `--dev /dev` - Provides access to device files
- `--proc /proc` - Provides access to /proc filesystem
- `-- debuild -b -uc -us` - Runs the Debian build process

#### Creating resolv.conf

If you don't have a `resolv.conf` file, you can create a minimal one:

```bash
echo "nameserver 127.0.0.1" > resolv.conf
```

Or copy your system's resolv.conf:

```bash
cp /etc/resolv.conf ./resolv.conf
```

Note: The `resolv.conf` file is used to prevent DNS resolution attempts during the build. Even with a non-functional nameserver, the build will work because all dependencies are vendored and no network access is required.

#### How It Works

The build process uses Debian's `dh-cargo` which:
1. Sets up a vendored cargo registry from `/usr/share/cargo/registry`
2. Creates a `config.toml` in `debian/cargo_home/` that replaces crates.io with the local registry
3. Pre-generates cargo metadata for cbindgen to avoid network access
4. Uses cbindgen with the pre-generated metadata to create C headers

All Rust dependencies are provided by Debian packages (e.g., `librust-libc-dev`, `librust-nix-dev`, etc.), so no network access to crates.io is needed.

## Package Structure

- `block-device-id` - Command-line tool
- `libblockdeviceid1` - Shared library (C-compatible)
- `libblockdeviceid-dev` - Development files (headers, examples)
- `librust-block-device-id-dev` - Rust source package

## Usage

### Command Line

```bash
block-device-id /dev/mmcblk0
```

### C API

See `examples/get_device_id.c` for a complete example.

```c
#include <block_device_id.h>

char id[129];
int ret = bdi_get_id("/dev/nvme0n1", id, sizeof(id));
if (ret == 0) {
    printf("Device ID: %s\n", id);
}
```

## License

BSD-3-Clause - See LICENSE file for details.
