//! CLI tool for retrieving block device identifiers.

use block_device_id::{get_id_by_path, Error};
use clap::Parser;
use std::process::ExitCode;

include!("../cli.rs");

fn main() -> ExitCode {
    let args = Args::parse();

    match get_id_by_path(&args.device) {
        Ok(id) => {
            print!("{}", id.id.to_string_lossy());
            ExitCode::from(0)
        }
        Err(e) => {
            let msg = match &e {
                Error::NotABlockDevice => "not a block device",
                Error::UnsupportedDevice => "unsupported device type (only MMC and NVMe are supported)",
                Error::NoIdentifierAvailable => "no innate identifier available",
                Error::IdentifierTruncated => "identifier too large (would be truncated)",
                Error::Io(io_err) => {
                    eprintln!("Error: {}: {}", args.device.display(), io_err);
                    return ExitCode::from(1);
                }
            };
            eprintln!("Error: {}: {}", args.device.display(), msg);
            ExitCode::from(1)
        }
    }
}
